#include <yplatform/log.h>
#include <yplatform/find.h>
#include <ymod_webserver/server.h>
#include <furita/pq/pq.hpp>
#include "ping_handler.h"
#include "preview_handler.h"
#include "apply_handler.h"
#include "bw_handler.h"
#include "bw_add_handler.h"
#include "bw_remove_handler.h"
#include "list_handler.h"
#include "list_batch_handler.h"
#include "remove_handler.h"
#include "enable_handler.h"
#include "order_handler.h"
#include "edit_handler.h"
#include "verify_handler.h"
#include "domain_rules_set_handler.h"
#include "domain_rules_get_handler.h"
#include "impl.h"
#include "util.h"

#include <tvm_guard/module.h>

namespace furita {
namespace api {

impl::impl()
{
    L_(info) << "furita_api task instantiated";
}

impl::~impl()
{
    L_(info) << "furita_api task destroyed";
}

void impl::init(const yplatform::ptree& conf)
{
    bbEnv = ConvertBBEnvironmentToEnum(conf.get<std::string>("blackbox_environment"));
}

void impl::fini()
{}

template <class TFunctor>
class TvmTicketsChecker {
public:
    explicit TvmTicketsChecker(TFunctor&& f, ymod_tvm::tvm2::blackbox_env bbEnv)
        : f(std::forward<TFunctor>(f))
        , bbEnv(bbEnv) {
    }

    void operator()(ymod_webserver::http::stream_ptr stream) {
        const auto error = CheckTvmTickets(stream, bbEnv);
        if (!error) {
            f(std::move(stream));
        } else {
            stream->result(ymod_webserver::codes::unauthorized, *error);
        }
    }

private:
    using TFunctorDecayed = std::decay_t<TFunctor>;

    TFunctorDecayed f;
    const ymod_tvm::tvm2::blackbox_env bbEnv;
};

struct HandlerWrap {
public:
    explicit HandlerWrap(std::shared_ptr<CommonHandler> handlerPtr)
        : handlerPtr(handlerPtr)
    {}

    void operator()(ymod_webserver::http::stream_ptr stream) {
        TContextPtr ctx = std::make_shared<TContext>((stream->ctx())->uniq_id());
        handlerPtr->execute(stream, ctx);
    }

private:
    std::shared_ptr<CommonHandler> handlerPtr;
};

void impl::start() {
    auto server = yplatform::find<ymod_webserver::server>("web_server");
    auto tvmGuard = yplatform::find<tvm_guard::Module>("tvm_guard");
    const std::string name = "main";    // endpoint name
    auto bind = [&name, server, bbEnv=bbEnv](const std::string& path, std::shared_ptr<CommonHandler> handler) {
        return server->bind(name, {path}, TvmTicketsChecker{HandlerWrap{std::move(handler)}, bbEnv});
    };

    bind("/ping", std::make_shared<PingHandler>());
    bind("/api/preview.json", std::make_shared<PreviewHandler>());
    bind("/api/apply.json", std::make_shared<ApplyHandler>());

    bind("/api/list.json", std::make_shared<ListHandler>());
    bind("/api/multi_list", std::make_shared<ListBatchHandler>());
    bind("/api/order.json", std::make_shared<OrderHandler>());
    bind("/api/edit.json", std::make_shared<EditHandler>());
    bind("/api/remove.json", std::make_shared<RemoveHandler>());
    bind("/api/enable.json", std::make_shared<EnableHandler>());
    bind("/api/verify.json", std::make_shared<VerifyHandler>());

    bind("/api/blacklist.json", std::make_shared<BWHandler>(rules::ListType::BLACK));
    bind("/api/whitelist.json", std::make_shared<BWHandler>(rules::ListType::WHITE));
    bind("/api/blackwhitelist", std::make_shared<BWHandler>(rules::ListType::ALL));
    bind("/api/blacklist_add.json", std::make_shared<BWAddHandler>(rules::ListType::BLACK));
    bind("/api/whitelist_add.json", std::make_shared<BWAddHandler>(rules::ListType::WHITE));
    bind("/api/blacklist_remove.json", std::make_shared<BWRemoveHandler>(rules::ListType::BLACK));
    bind("/api/whitelist_remove.json", std::make_shared<BWRemoveHandler>(rules::ListType::WHITE));

    auto bindTvmGuard = [&name, server, tvmGuard](const std::string& path, std::shared_ptr<CommonHandler> handler) {
        tvmGuard->bind(*server, name, {path}, HandlerWrap{std::move(handler)});
    };
    bindTvmGuard("/v1/domain/rules/set", std::make_shared<TDomainRulesSetHandler>());
    bindTvmGuard("/v1/domain/rules/get", std::make_shared<TDomainRulesGetHandler>());
}

}   // namespace api
}   // namespace furita

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(furita::api::impl)
