#pragma once

#include <string>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#include <boost/utility/enable_if.hpp>
#include <contrib/libs/yajl/api/yajl_gen.h>

class json_writer
{
public:
    explicit json_writer()
    {
        if ((m_generator = yajl_gen_alloc(NULL)) == NULL)
            throw std::runtime_error("cannot create JSON generator");
    }

    ~json_writer()
    {
        yajl_gen_free(m_generator);
    }

    json_writer & begin_object()
    {
        check_status(yajl_gen_map_open(m_generator));
        return *this;
    }

    json_writer & begin_object(const std::string &name)
    {
        write_string(name);
        return begin_object();
    }

    json_writer & end_object()
    {
        check_status(yajl_gen_map_close(m_generator));
        return *this;
    }

    json_writer & begin_array()
    {
        check_status(yajl_gen_array_open(m_generator));
        return *this;
    }

    json_writer & begin_array(const std::string &name)
    {
        write_string(name);
        return begin_array();
    }

    json_writer & end_array()
    {
        check_status(yajl_gen_array_close(m_generator));
        return *this;
    }

    json_writer & add_member(const std::string &value)
    {
        write_string(value);
        return *this;
    }

    json_writer & add_member(bool value)
    {
        check_status(yajl_gen_bool(m_generator, value));
        return *this;
    }

    template <typename T>
    json_writer & add_member(T value)
    {
        generic_writer<T> writer(*this, value);
        return *this;
    }

    template <typename T>
    json_writer & add_member(const std::string &name, T value)
    {
        write_string(name);
        return add_member(value);
    }

    json_writer & close()
    {
        check_status(yajl_gen_get_buf(m_generator, &m_buffer, &m_buffer_length));
        return *this;
    }

    std::size_t size() const
    {
        if (m_buffer == NULL)
            throw std::runtime_error("document is not closed");
        return m_buffer_length;
    }

    const char * data() const
    {
        if (m_buffer == NULL)
            throw std::runtime_error("document is not closed");
        return reinterpret_cast<const char *>(m_buffer);
    }

private:
    void write_string(const std::string &value)
    {
        std::basic_string<unsigned char> buffer(value.begin(), value.end());
        check_status(yajl_gen_string(m_generator, buffer.c_str(), buffer.length()));
    }

    void check_status(yajl_gen_status status) const
    {
        if (status != yajl_gen_status_ok)
            throw std::runtime_error("JSON generator error");
    }

    template <typename T, typename Enable = void>
    struct generic_writer
    {
        generic_writer(json_writer &writer, T value)
        {
            writer.add_member(boost::lexical_cast<std::string>(value));
        }
    };

    template <typename T>
    struct generic_writer<T, typename boost::enable_if<boost::is_integral<T> >::type>
    {
        generic_writer(json_writer &writer, T value)
        {
            writer.check_status(yajl_gen_integer(writer.m_generator, value));
        }
    };

    template <typename T>
    struct generic_writer<T, typename boost::enable_if<boost::is_float<T> >::type>
    {
        generic_writer(json_writer &writer, T value)
        {
            writer.check_status(yajl_gen_double(writer.m_generator, value));
        }
    };

    yajl_gen m_generator;

    const unsigned char *m_buffer;
    std::size_t m_buffer_length;
};

