#include "util.h"
#include "buffer_chunk.h"
#include "list_batch_handler.h"
#include "../processor/make_search_helper.hpp"

#include <yplatform/find.h>
#include <library/cpp/scheme/scheme.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>

#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/range/algorithm.hpp>

using namespace furita;
using namespace furita::api;
using namespace furita::processor::msq;

struct ListBatchHandler::Data {
    Data() = default;

    HttpStreamPtr stream;
    std::vector<uint64_t> uids;
    sharpei::client::Mode sharpeiMode;
};


void ListBatchHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto arg = std::make_shared<Data>();
    try {
        const auto& params = stream->request()->url.params;
        arg->stream = stream;
        const auto& uids_str = parameterValue(params, "uids", std::string());

        if (uids_str.empty()) {
            promise<boost::unordered_map<ui64, furita::pq::RulesResult>> p;
            p.set(boost::unordered_map<ui64, furita::pq::RulesResult>());
            future<boost::unordered_map<ui64, furita::pq::RulesResult>> f(p);
            return HandleResult(arg, f, ctx);
        }

        std::vector<std::string> uids;
        boost::split(uids, uids_str, boost::is_any_of(","));
        std::transform(uids.begin(), uids.end(), std::back_inserter(arg->uids),
                       [](const std::string& str) {
                            try {
                                return boost::lexical_cast<uint64_t>(str);
                            } catch(boost::bad_lexical_cast&) {
                                throw std::runtime_error("Error while converting list param: " + str + " to uint64_t");
                            }
                       });

        auto m_master = parameterValue(params, "master", false);
        arg->sharpeiMode = m_master
                               ? sharpei::client::Mode::WriteOnly
                               : sharpei::client::Mode::ReadWrite;
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="multi_list operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    auto pq = yplatform::find<pq::pq>("furita_pq");
    auto f = pq->get_rules_multiuser(ctx, arg->uids, arg->sharpeiMode);

    f.add_callback(boost::bind(&ListBatchHandler::HandleResult, this, arg, f, ctx));
}

void ListBatchHandler::HandleResult(const DataPtr args, future<boost::unordered_map<uint64_t, furita::pq::RulesResult>> f, TContextPtr ctx) const {
    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);
    json_writer& w = buffer->m_writer;
    w.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    const auto users_with_rules = f.get();
    w.begin_array("users");
    for (auto it = users_with_rules.begin(); it != users_with_rules.end(); it++) {
        auto uid = it->first;
        auto rules_result = it->second;
        w.begin_object();
        w.add_member("uid", uid);
        if (rules_result.which() == 0) {
            w.add_member("result", "ok");
            w.begin_array("rules");
            auto rules = boost::get<rules::rule_list_ptr>(rules_result);
            furita::rules_helpers::hackListPriority(rules);
            for (const auto& rule : *rules) {
                w.begin_object();
                w.add_member("id", std::to_string(rule->id));
                w.add_member("name", rule->name);
                w.add_member("priority", rule->prio);
                w.add_member("stop", rule->stop);
                w.add_member("enabled", rule->enabled);
                w.add_member("created", rule->created);
                w.add_member("type", rule->type);
                w.add_member("query", getQueryString(rule, args, std::to_string(uid), ctx));

                w.begin_array("actions");
                for (const auto& action : *rule->actions) {
                    w.begin_object().add_member("type", action->oper);
                    if (!action->param.empty()) {
                        w.add_member("parameter", action->param);
                    }
                    w.add_member("verified", action->verified);
                    w.end_object();
                }
                w.end_array();

                w.begin_array("conditions");
                for (const auto& condition : *rule->conditions) {
                    w.begin_object();
                    w.add_member("field_type", condition->field_type);
                    w.add_member("div", condition->field);
                    if (!condition->pattern.empty()) {
                        w.add_member("pattern", condition->pattern);
                    }
                    w.add_member("oper", condition->oper * 2 + (condition->neg ? 2 : 1));
                    w.add_member("link", condition->link);
                    w.end_object();
                }
                w.end_array().end_object();
            }
            w.end_array();
        } else {
            w.add_member("result", "error");
            w.add_member("message", boost::get<mail_errors::error_code>(rules_result).message());
        }
        w.end_object();
    }
    w.end_array().end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(w.size())->send(buffer);
}

std::string ListBatchHandler::getQueryString(const rules::rule_ptr& rule, const DataPtr /*args*/, const std::string& uid, TContextPtr ctx) const {
    std::string query;
    try {
        query = make_search_query(rule, uid);
    } catch (const std::exception& e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="can't generate search query: status=error")
    }
    return query;
}
