#include <boost/unordered_map.hpp>
#include <yplatform/find.h>
#include <furita/pq/pq.hpp>
#include <furita/common/logger.h>
#include "util.h"
#include "buffer_chunk.h"
#include "list_handler.h"
#include "../processor/make_search_helper.hpp"

using namespace furita;
using namespace furita::api;
using namespace furita::processor::msq;

struct ListHandler::Data {
    HttpStreamPtr stream;

    uint64_t uid;
    std::string m_id;
    bool m_detailed, m_master;
    rules_helpers::OptionalRuleType m_type;

    rules::rule_list_ptr m_rules;
    boost::unordered_map<uint64_t, rules::rule_ptr> m_rules_map;
    sharpei::client::Mode mode;
};

void ListHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Data>();
    args->stream = stream;

    try {
        const auto& params = stream->request()->url.params;
        args->uid = parameterValue<uint64_t>(params, "uid");
        args->m_id = parameterValue(params, "id", std::string());
        args->m_detailed = parameterValue(params, "detailed", false);
        args->m_type = rules_helpers::RuleType::create(parameterValue(params, "type", std::string()));
        args->m_master = parameterValue(params, "master", false);
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="list operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        args->mode = args->m_master
                           ? sharpei::client::Mode::WriteOnly
                           : sharpei::client::Mode::ReadWrite;
        future<rules::rule_list_ptr> futureRules;
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid, resolverFactory, args->mode);

        if (!args->m_id.empty()) {
            futureRules = pq->get_rule(executor, args->uid, std::stoull(args->m_id));
        } else if (args->m_type) {
            futureRules = pq->get_rules_by_type(executor, args->uid, *args->m_type);
        } else {
            futureRules = pq->get_all_rules(executor, args->uid);
        }
        futureRules.add_callback(boost::bind(&ListHandler::handleRules, this, args, executor, futureRules, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="list operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void ListHandler::handleRules(DataPtr args, pgg::RequestExecutor& executor,
                              future<rules::rule_list_ptr> f_rules, TContextPtr ctx) const {
    if (f_rules.has_exception()) {
        return handleError(args->stream, "list", f_rules);
    }
    args->m_rules = f_rules.get();

    for (const auto& rule: *args->m_rules) {
        args->m_rules_map.emplace(rule->id, rule);
    }

    auto pq = yplatform::find<pq::pq>("furita_pq");
    future<rules::action_list_ptr> futureActions;
    if (!args->m_id.empty()) {
        futureActions = pq->get_rule_actions(executor, args->uid, std::stoull(args->m_id));
    } else {
        futureActions = pq->get_all_actions(executor, args->uid);
    }

    futureActions.add_callback(boost::bind(&ListHandler::handleActions, this, args, executor, futureActions, ctx));
}

void ListHandler::handleActions(DataPtr args, pgg::RequestExecutor& executor, future<rules::action_list_ptr> f_actions, TContextPtr ctx) const {
    if (f_actions.has_exception()) {
        return handleError(args->stream, "list", f_actions);
    }

    for(const auto& a : *f_actions.get()) {
        const auto i = args->m_rules_map.find(a->rule_id);
        if (i != args->m_rules_map.end()) {
            i->second->actions->push_back(a);
        }
    }

    auto pq = yplatform::find<pq::pq>("furita_pq");
    if (args->m_detailed) {
        auto futureResult = !args->m_id.empty()
            ? pq->get_rule_conditions(executor, args->uid, std::stoull(args->m_id))
            : pq->get_all_conditions(executor, args->uid);
        futureResult.add_callback(
            boost::bind(&ListHandler::handleConditions, this, args, futureResult, ctx)
        );
    } else {
        handleResult(args, ctx);
    }
}

void ListHandler::handleConditions(DataPtr args, future<rules::condition_list_ptr> r, TContextPtr ctx) const {
    if (r.has_exception()) {
        return handleError(args->stream, "list", r);
    }
    for (const auto& condition: *r.get()) {
        auto iter = args->m_rules_map.find(condition->rule_id);
        if (iter != args->m_rules_map.end()) {
            iter->second->conditions->push_back(condition);
        }
    }
    handleResult(args, ctx);
}

void ListHandler::handleResult(const DataPtr& args, TContextPtr ctx) const {
    args->m_rules_map.clear();

    FURITA_LOG_NOTICE(ctx, logdog::message="list operation finished: status=ok")

    furita::rules_helpers::hackListPriority(args->m_rules);

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);
    json_writer &w = buffer->m_writer;
    w.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    w.begin_array("rules");
    for (const auto& rule: *args->m_rules) {
        w.begin_object();
        w.add_member("id", std::to_string(rule->id));
        w.add_member("name", rule->name);
        w.add_member("priority", rule->prio);
        w.add_member("stop", rule->stop);
        w.add_member("enabled", rule->enabled);
        w.add_member("created", rule->created);
        w.add_member("type", rule->type);

        w.begin_array("actions");
        for (const auto& action: *rule->actions) {
            w.begin_object().add_member("type", action->oper);
            if (!action->param.empty()) {
                w.add_member("parameter", action->param);
            }
            w.add_member("verified", action->verified);
            w.end_object();
        }
        w.end_array();

        w.begin_array("conditions");
        for (const auto& condition: *rule->conditions) {
            w.begin_object();
            w.add_member("field_type", condition->field_type);
            w.add_member("div", condition->field);
            if (!condition->pattern.empty()) {
                w.add_member("pattern", condition->pattern);
            }
            w.add_member("oper", condition->oper * 2 + (condition->neg ? 2 : 1));
            w.add_member("link", static_cast<int>(condition->link));
            w.end_object();
        }
        w.end_array();

        if (args->m_detailed) {
            const std::string query = getQueryString(rule, args, ctx);
            w.add_member("query", query);
        }

        w.end_object();
    }
    w.end_array().end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(w.size())->send(buffer);
}

std::string ListHandler::getQueryString(const rules::rule_ptr& rule, const DataPtr& args, TContextPtr ctx) const {
    std::string query;
    try {
        query = make_search_query(rule, std::to_string(args->uid));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="can't generate search query: status=error")
    }
    return query;
}
