#include <algorithm>
#include <boost/tokenizer.hpp>
#include <yplatform/find.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>
#include "util.h"
#include "buffer_chunk.h"
#include "order_handler.h"

namespace furita {
namespace api {

struct OrderHandler::Arguments {
    HttpStreamPtr stream;
    uint64_t uid_;
    std::vector<uint64_t> ids_;
};

void OrderHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid_ = parameterValue<uint64_t>(params, "uid");
        const std::string& list = parameterValue(params, "list");

        typedef boost::char_separator<char> separator;
        boost::tokenizer<separator> tokens(list, separator(","));
        std::transform(tokens.begin(), tokens.end(), std::back_inserter(args->ids_),
                   [](const std::string& str) {
                        try {
                            return boost::lexical_cast<uint64_t>(str);
                        }  catch(boost::bad_lexical_cast&) {
                            throw std::runtime_error("Error while convertind list param to uint64_t");
                        }
                   });

        if (args->ids_.empty()) {
            throw std::runtime_error("Rule list is empty");
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="order operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid_, resolverFactory);

        auto futureResult = pq->order_rule(executor, args->uid_, args->ids_);
        futureResult.add_callback(boost::bind(&OrderHandler::handleResult, this, args, futureResult, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="order operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void OrderHandler::handleResult(ArgumentsPtr args, future<bool> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "order", result);
    }

    if (result.get() == false) {
        return handleFail(args->stream, "Given list is incorrect");
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="order operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
