#include <boost/range/iterator_range.hpp>
#include <yplatform/find.h>
#include <src/processor/http_client.hpp>
#include "util.h"
#include "json_writer.h"
#include "buffer_chunk.h"
#include "preview_handler.h"
#include <furita/common/http_headers.h>
#include <furita/common/logger.h>
#include <furita/common/rule_helper.hpp>

namespace furita {
namespace api {

void PreviewHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    uint64_t uid;
    boost::optional<uint64_t> id;
    std::string remote_ip, tvm;
    unsigned int length, offset;
    rules::rule_ptr rule = boost::make_shared<rules::rule>();
    ymod_httpclient::headers_dict headers;
    try {
        const auto& params = stream->request()->url.params;
        if (params.find("id") != params.end() && !parameterValue<std::string>(params, "id").empty()) {
            id = parameterValue<uint64_t>(params, "id");
        }
        uid = parameterValue<uint64_t>(params, "uid");
        length = parameterValue<unsigned int>(params, "length", 10);
        offset = parameterValue<unsigned int>(params, "offset", 0);
        remote_ip = parameterValue(params, "remote_ip", std::string());
        auto& request_headers = stream->request()->headers;
        tvm = request_headers[HttpHeaderNames::ticket];
        if (auto it = request_headers.find(HttpHeaderNames::tvmUserTicket); it != request_headers.end()) {
            headers.emplace(HttpHeaderNames::tvmUserTicket, it->second); // we've already checked it
        }

        if (!id) {
            rules::condition::link_type logic = parameterValue<int>(params, "logic", rules::condition::link_type::OR)
                                                ? rules::condition::link_type::AND
                                                : rules::condition::link_type::OR; // 0 - OR, 1 - AND
            int attachment = parameterValue<int>(params, "attachment", rules_helpers::AttachmentParamType::ALL); // 0 - all, 1 - with attaches, 2 - no attaches

            rules::condition_list_ptr conditions(new rules::condition_list);
            if (attachment != rules_helpers::AttachmentParamType::ALL) {
                conditions->push_back(boost::make_shared<rules::condition>(
                    "att", "", rules::condition::link_type::AND, rules::condition::oper_type::MATCHES,
                    (attachment != rules_helpers::AttachmentParamType::WITH_ATTACHES))
                );
            }
            auto divs = params.equal_range("field1"); // field
            auto operations = params.equal_range("field2"); // logic
            auto patterns = params.equal_range("field3"); // pattern

            if (conditions->empty() &&
                ((divs.first == divs.second) ||
                (operations.first == operations.second) ||
                (patterns.first == patterns.second)))
            {
                throw std::runtime_error("No enough parameters");
            }

            auto iterDiv = divs.first;
            auto iterOp = operations.first;
            auto iterPattern = patterns.first;
            for (; iterDiv != divs.second &&
                   iterOp != operations.second &&
                   iterPattern != patterns.second;
                ++iterDiv, ++iterOp, ++iterPattern)
            {
                rules_helpers::ConditionOper op(iterOp->second);
                rules::condition_ptr c(new rules::condition);
                c->field_type = rules_helpers::findFieldType(iterDiv->second);
                c->field = iterDiv->second;
                c->pattern = iterPattern->second;
                c->link = logic;
                c->oper = op.toRuleOperType();
                c->neg = op.negative();

                if (iterDiv->second == "type" && op.toRuleOperType() != rules::condition::oper_type::MATCHES) {
                    throw std::runtime_error("Message type's operation supports only MATCHES value");
                }
                if (op.toRuleOperType() == rules::condition::oper_type::EXISTS &&
                    !rules_helpers::acceptableForExistsCondition(iterDiv->second)) {
                    throw std::runtime_error("EXISTS operation is acceptable only for custom headers");
                }

                conditions->push_back(c);
            }
            if (iterDiv != divs.second ||
                iterOp != operations.second ||
                iterPattern != patterns.second)
            {
                throw std::runtime_error("No enough parameters");
            }

            rule->conditions.swap(conditions);
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="preview operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto processor = yplatform::find<processor::processor>("furita_processor");
        auto result = id ?
            processor->preview(ctx, uid, id.get(), length, offset, remote_ip, tvm, headers) :
            processor->preview(ctx, uid, rule, length, offset, remote_ip, tvm, headers);
        result.add_callback(boost::bind(&PreviewHandler::handleResult, this, stream, result, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="preview operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void PreviewHandler::handleResult(HttpStreamPtr stream, future<std::vector<std::string>> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        std::string description = exception_description(result);
        FURITA_LOG_ERROR(ctx, logdog::message="preview operation finished: status=error, report='" + description + "'")
        return _handleError(stream, description);
    }

    auto messages = result.get();

    FURITA_LOG_NOTICE(ctx, logdog::message="preview operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object();
    writer.add_member("session", stream->ctx()->uniq_id()).add_member("status", "ok");
    writer.add_member("size", messages.size()).begin_array("messages");
    for (const auto& value: messages) {
        writer.add_member(value);
    }
    writer.end_array().end_object().close();

    stream->set_code(ymod_webserver::codes::ok);
    stream->set_content_type("application", "json");
    stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
