#include <boost/algorithm/string.hpp>
#include <yplatform/find.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>
#include "util.h"
#include "buffer_chunk.h"
#include "remove_handler.h"

namespace furita {
namespace api {

struct RemoveHandler::Arguments {
    HttpStreamPtr stream;
    uint64_t uid_;
    std::vector<uint64_t> ids_;
};

void RemoveHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid_ = parameterValue<uint64_t>(params, "uid");

        auto ids = params.equal_range("id");
        for (auto id = ids.first; id != ids.second; ++id) {
            if (!id->second.empty()) {
                args->ids_.push_back(std::stoull(id->second));
            }
        }
        if (args->ids_.empty()) {
            throw std::runtime_error("at least one 'id' should be defined");
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="remove operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid_, resolverFactory);

        auto futureResult = pq->remove_rules(executor, args->uid_, args->ids_);
        futureResult.add_callback(boost::bind(&RemoveHandler::handleResult, this, args, futureResult, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="remove operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void RemoveHandler::handleResult(ArgumentsPtr args, future<void> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "remove", result);
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="remove operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
