#include <boost/spirit/include/qi.hpp>
#include "util.h"
#include <furita/common/http_headers.h>
#include <util/system/compiler.h>
#include <yplatform/json.h>

namespace furita {

template <typename Iterator>
class address_grammar : public boost::spirit::qi::grammar<Iterator>
{
public:
    explicit address_grammar() : address_grammar::base_type(m_mailbox)
    {
        using boost::spirit::qi::alnum;
        using boost::spirit::qi::blank;
        using boost::spirit::qi::char_;

        m_atext = alnum | char_("!#$%&'*+/=?^_`{|}~-");
        m_dot_atom_text = +m_atext >> *('.' >> +m_atext);

        m_quoted_content = (char_ - char_("\\\"")) | ('\\' >> char_);
        m_quoted_string = '"' >> *(-blank >> m_quoted_content) >> -blank >> '"';

        m_subdomain = alnum >> *(alnum | '-');
        m_domain = m_subdomain >> *('.' >> m_subdomain);

        m_mailbox = (m_dot_atom_text | m_quoted_string) >> '@' >> (m_domain | ('[' >> m_dot_atom_text >> ']'));
    }

private:
    typedef boost::spirit::qi::rule<Iterator> rule;
    rule m_atext, m_dot_atom_text;
    rule m_quoted_content, m_quoted_string;
    rule m_subdomain, m_domain;
    rule m_mailbox;
};

bool check_address(const std::string &address)
{
    std::string::const_iterator begin = address.begin(), end = address.end();
    address_grammar<std::string::const_iterator> grammar;
    bool r = boost::spirit::qi::parse(begin, end, grammar);
    return r && begin == end;
}

ymod_tvm::tvm2::blackbox_env ConvertBBEnvironmentToEnum(const std::string& env) {
    static const std::map<std::string, ymod_tvm::tvm2::blackbox_env> envsByName = {
        {"blackbox", TA_BE_PROD},
        {"blackbox-test", TA_BE_TEST},
        {"blackbox-corp", TA_BE_PROD_YATEAM},
        {"blackbox-corp-test", TA_BE_TEST_YATEAM},
        {"blackbox-stress", TA_BE_STRESS},
        {"blackbox-mimino", TA_BE_PROD}};
    return envsByName.at(env);
}

std::optional<TErrorMsg> CheckTvmTickets(ymod_webserver::http::stream_ptr stream, ymod_tvm::tvm2::blackbox_env bbEnv) {
    const auto& headers = stream->request()->headers;
    if (auto it = headers.find(furita::HttpHeaderNames::tvmUserTicket); it != headers.end()) {
        auto tvm = yplatform::find<ymod_tvm::tvm2_module>("tvm");
        const auto error = tvm->check_user_ticket(stream->ctx(), bbEnv, it->second);
        if (Y_UNLIKELY(error)) {
            return std::make_optional(error.message());
        }
    }
    return std::nullopt;
}

namespace api {

void ResponseMessage(HttpStreamPtr stream, ymod_webserver::codes::code code, std::string msg) {
    yplatform::json_value val;
    val["message"] = std::move(msg);
    stream->set_code(std::move(code));
    stream->set_content_type("application", "json");
    stream->result_body(val.stringify());
}

}

}
