#include <openssl/evp.h>
#include <library/cpp/openssl/holders/evp.h>
#include <yplatform/find.h>
#include <yplatform/encoding/base64.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>
#include "util.h"
#include "buffer_chunk.h"
#include "verify_handler.h"

namespace furita {
namespace api {
constexpr int VERIFY_HANDLER_PARAMS_COUNT = 4;

namespace
{
    void decrypt(const std::vector<unsigned char> &in, std::string &out,
        const std::string &key, const std::string &iv)
    {
        int length = 0, final_length = 0;
        out.resize(in.size() + 1024);

        NOpenSSL::TEvpCipherCtx ctx;
        EVP_DecryptInit_ex(ctx, EVP_bf_cbc(), NULL,
            reinterpret_cast<const unsigned char *>(key.c_str()),
            reinterpret_cast<const unsigned char*>(iv.c_str()));

        if (!EVP_DecryptUpdate(ctx, reinterpret_cast<unsigned char *>(&out[0]),
            &length, in.data(), static_cast<int>(in.size())))
        {
            throw std::runtime_error("Failed to decrypt token");
        }

        if (!EVP_DecryptFinal_ex(ctx,
            reinterpret_cast<unsigned char *>(&out[length]), &final_length))
        {
            throw std::runtime_error("Failed to decrypt token");
        }

        out.resize(length + final_length);
    }

    template <typename T, typename Range>
    inline std::vector<T> make_vector(const Range &range) {
        return std::vector<T>{range.begin(), range.end()};
    }

    void decode(const std::string& e, uint64_t& uid,
        uint64_t& id, std::string& from)
    {
        std::string in;
        const std::string key("FilterRequestKey"), iv("Anything");

        decrypt(make_vector<unsigned char>(yplatform::base64_decode(e)), in, key, iv);

        typedef boost::char_separator<char> separator;
        boost::tokenizer<separator> tokens(in,
            separator(":", "", boost::keep_empty_tokens));
        std::vector<std::string> values(tokens.begin(), tokens.end());

        if (values.size() != VERIFY_HANDLER_PARAMS_COUNT) {
            throw std::runtime_error("Wrong fields in the token");
        }

        if (!boost::conversion::try_lexical_convert(values[1], id)) {
            throw std::runtime_error("Can't cast " + values[1] + " to id");
        }
        from = values[2];
        if (!boost::conversion::try_lexical_convert(values[3], uid)) {
            throw std::runtime_error("Can't cast " + values[3] + " to uid");
        }


        if (time(NULL) > boost::lexical_cast<int>(values[0])) {
            throw std::runtime_error("Token expired");
        }
    }
}

struct VerifyHandler::Data {
    HttpStreamPtr stream;
    uint64_t uid_, id_;
    std::string from_, afrom_, param_;
};

void VerifyHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto arg = std::make_shared<Data>();
    arg->stream = stream;

    try {
        const auto& params = stream->request()->url.params;
        decode(parameterValue(params, "e"), arg->uid_, arg->id_, arg->from_);

        arg->afrom_ = parameterValue(params, "from", std::string());
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="verify operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, arg->uid_, resolverFactory);

        auto futureResult = pq->get_action(executor, arg->uid_, arg->id_);
        futureResult.add_callback(
            boost::bind(&VerifyHandler::handleAction, this, arg, executor, futureResult, ctx)
        );
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="verify operation finished: status=error")
        return _handleError(stream, "Internal server error");
    }
}


void VerifyHandler::handleAction(DataPtr arg, pgg::RequestExecutor& executor, future<rules::action_list_ptr> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(arg->stream, "verify", result);
    }

    try {
        if (result.get()->empty()) {
            throw std::runtime_error("Failed to find action");
        }

        rules::action_ptr a = *result.get()->begin();
        if (a->param.empty()) {
            throw std::runtime_error("Parameter does not exist");
        }
        if (a->verified) {
            throw std::runtime_error("Parameter already verified");
        }

        arg->param_ = a->param;

        if (arg->afrom_.empty()) {
            auto pq = yplatform::find<pq::pq>("furita_pq");
            auto futureResult = pq->
                verify_rule(executor, arg->uid_, arg->id_);
            futureResult.add_callback(
                boost::bind(&VerifyHandler::handleVerify, this, arg, futureResult, ctx)
            );
        } else {
            handleResult(arg, ctx);
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="verify operation finished: status=error")
        _handleError(arg->stream, e.what());
    }
}

void VerifyHandler::handleVerify(DataPtr arg, future<void> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(arg->stream, "verify", result);
    }
    handleResult(arg, ctx);
}

void VerifyHandler::handleResult(const DataPtr& arg, TContextPtr ctx) const {
    FURITA_LOG_NOTICE(ctx, logdog::message="verify operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", arg->stream->ctx()->uniq_id());
    writer.add_member("fwd_from", arg->from_);
    writer.add_member("fwd_to", arg->param_);
    writer.end_object().close();

    arg->stream->set_code(ymod_webserver::codes::ok);
    arg->stream->set_content_type("application", "json");
    arg->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
