#pragma once

#include <boost/system/error_code.hpp>

namespace furita::blackbox {

enum class EError {
    Ok = 0,
    Unknown,
    NotFound,
    MultipleUsersFound,
    EmptyOrgId,
    InvalidResponse,
    TvmServiceTicketError,
};

}

namespace boost::system {

template <>
struct is_error_code_enum<furita::blackbox::EError> : std::true_type {};

}

namespace furita::blackbox {

class TErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "furita_tupita_converter";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::InvalidResponse:
                return "Invalid response";
            case EError::NotFound:
                return "User not found";
            case EError::MultipleUsersFound:
                return "Multiple users found";
            case EError::EmptyOrgId:
                return "Empty org ID";
            case EError::TvmServiceTicketError:
                return "Error obtaining TVM ticket";
            case EError::Unknown:
                break;
        }
        return "Unknown error code: " + std::to_string(value);
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

}
