#include "http_client.h"

#include <mail/furita/include/furita/common/tvm.hpp>
#include <yamail/data/deserialization/yajl.h>

#include <boost/asio/yield.hpp>

namespace furita::blackbox {

const std::string BLACKBOX_SERVICE_NAME{"blackbox"};

void TBlackBoxRequestRunner::Run(TErrorCode ec, const yhttp::response& response)
{
    reenter(Coro) {
        for (; EmailsCurr != Request.Emails.end(); ++EmailsCurr) {
            ErrorMessage.clear();
            yield {
                auto httpRequest = ymod_httpclient::request::GET(MakeUrl(*EmailsCurr));
                return ClusterCall->async_run(Ctx->CreateTaskContext(), std::move(httpRequest),
                    [this, self = shared_from_this()] (auto ec, auto response) {
                        boost::asio::post(IoContext, std::bind(&TBlackBoxRequestRunner::Run, self, ec, response));
                    }
                );
            }
            if (ec) {
                ErrorCode = ec;
                yield break;
            }

            OnSingleResponse(response.body);
        }
        if (ErrorCode) {
            Response.clear();
        }
    }

    if (Coro.is_complete()) {
        Callback(ErrorCode, {.OrgIds = Response, .ErrorMessage = ErrorMessage});
    }
}

void TBlackBoxRequestRunner::OnSingleResponse(const std::string& responseStr) {
    using namespace blackbox::reflection;

    auto error = blackbox::EError::Unknown;
    std::string orgId;
    std::string errorMessage;

    try {
        auto parsed = yamail::data::deserialization::fromJson<TResponse>(responseStr);
        if (parsed.Users.size() == 1) {
            orgId = parsed.Users[0].Attributes.OrgId;
            if (!orgId.empty()) {
                error = blackbox::EError::Ok;
            } else {
                error = blackbox::EError::EmptyOrgId;
            }
        } else if (parsed.Users.empty()) {
            error = blackbox::EError::NotFound;
        } else {
            error = blackbox::EError::MultipleUsersFound;
        }
    } catch (const std::exception& e) {
        FURITA_LOG_ERROR(Ctx, logdog::message="Exception was thrown", logdog::exception=e);
        error = blackbox::EError::InvalidResponse;
        errorMessage = e.what();
    }

    ErrorCode = error;
    ErrorMessage = errorMessage;
    Response.push_back(orgId);
}

std::string TBlackBoxRequestRunner::MakeUrl(const std::string& email) {
    std::string url{"/blackbox"};
    auto params = yhttp::url_encode({
        {"method", "userinfo"},
        {"login", email},
        {"attributes", "1031"},
        {"userip", "127.0.0.1"},
        {"format", "json"},
    });
    return url + params;
}

void TBlackBoxClient::DoRequest(TContextPtr ctx, TBlackBoxRequest request,
    TBlackBoxClientCallback clientCb)
{
    if (request.Emails.empty()) {
        return clientCb(TErrorCode{}, {});
    }

    auto runner = std::make_shared<TBlackBoxRequestRunner>(std::move(ctx), ClusterCall, IoContext, std::move(request), std::move(clientCb));
    runner->Run();
}

}
