#pragma once

#include "error_code.h"

#include <mail/butil/include/butil/http/headers.h>
#include <mail/furita/include/furita/common/context.h>
#include <mail/furita/include/furita/common/logger.h>
#include <yamail/data/reflection/reflection.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_tvm/module.h>

#include <boost/asio.hpp>
#include <boost/asio/io_context.hpp>

namespace furita::blackbox::reflection {
    struct TAttributes {
        std::string OrgId;
    };

    struct TUser {
        TAttributes Attributes;
    };

    struct TResponse {
        std::vector<TUser> Users;
    };
}

YREFLECTION_ADAPT_ADT(furita::blackbox::reflection::TAttributes,
    YREFLECTION_MEMBER_RENAMED(std::string, 1031, OrgId)
);

YREFLECTION_ADAPT_ADT(furita::blackbox::reflection::TUser,
    YREFLECTION_MEMBER_RENAMED(furita::blackbox::reflection::TAttributes, attributes, Attributes)
);

YREFLECTION_ADAPT_ADT(furita::blackbox::reflection::TResponse,
    YREFLECTION_MEMBER_RENAMED(std::vector<furita::blackbox::reflection::TUser>, users, Users)
);


namespace furita::blackbox {

struct TBlackBoxRequest {
    std::vector<std::string> Emails;
};

struct TBlackBoxResponse {
    std::vector<std::string> OrgIds;
    std::string ErrorMessage;
};

using TErrorCode = ::boost::system::error_code;
using TBlackBoxClientCallback = std::function<void(TErrorCode, const TBlackBoxResponse)>;
using TClusterCall = ymod_httpclient::cluster_call;
using TClusterCallPtr = std::shared_ptr<TClusterCall>;


class TBlackBoxRequestRunner : public std::enable_shared_from_this<TBlackBoxRequestRunner> {
public:
    TBlackBoxRequestRunner(TContextPtr ctx, TClusterCallPtr clusterCall,
        boost::asio::io_context& ioContext, TBlackBoxRequest request, TBlackBoxClientCallback clientCb)
            : Ctx(std::move(ctx))
            , ClusterCall(clusterCall)
            , IoContext(ioContext)
            , Callback(clientCb)
            , Request(std::move(request))
            , EmailsCurr(Request.Emails.begin())
    {
        Response.reserve(Request.Emails.size());
    }

    void Run(TErrorCode ec = {}, const yhttp::response& response = {});

private:
    void OnSingleResponse(const std::string& responseStr);
    static std::string MakeUrl(const std::string& email);

private:
    TContextPtr Ctx;
    TClusterCallPtr ClusterCall;
    boost::asio::io_context& IoContext;
    boost::asio::coroutine Coro;
    TBlackBoxClientCallback Callback;

    TBlackBoxRequest Request;
    std::vector<std::string>::const_iterator EmailsCurr;

    std::vector<std::string> Response;
    TErrorCode ErrorCode;
    std::string ErrorMessage;
};

struct IBlackBoxClient {
    virtual ~IBlackBoxClient() = default;
    virtual void DoRequest(TContextPtr ctx, TBlackBoxRequest request, TBlackBoxClientCallback clientCb) = 0;
};

class TBlackBoxClient : public IBlackBoxClient {
public:
    TBlackBoxClient(TClusterCallPtr clusterCall, boost::asio::io_context& ioContext)
        : ClusterCall(clusterCall)
        , IoContext(ioContext)
    {
    }
    void DoRequest(TContextPtr ctx, TBlackBoxRequest request, TBlackBoxClientCallback clientCb) override;

private:
    TClusterCallPtr ClusterCall;
    boost::asio::io_context& IoContext;
};

using TBlackBoxClientPtr = std::shared_ptr<IBlackBoxClient>;

}
