#include "impl.hpp"
#include "queries.hpp"
#include "multiuser_rules_handler.hpp"
#include "get_rules_handler.hpp"
#include "get_conditions_handler.hpp"
#include "get_actions_handler.hpp"
#include "bw_list_handler.hpp"
#include "edit_rule_handler.hpp"
#include "get_not_verified_params_handler.hpp"

#include <pgg/factory.h>
#include <pgg/service/uid_resolver.h>
#include <pgg/service/shard_resolver.h>
#include <yplatform/exception.h>
#include <yplatform/find.h>
#include <macs_pg/integration/pa_profiler.h>

#include <boost/unordered_map.hpp>
#include <boost/assign/list_of.hpp>

namespace furita {
namespace pq {
impl::impl() {
    L_(info) << "furita_pq task instantiated";
}

impl::~impl() {
    L_(info) << "furita_pq task destroyed";
}

void impl::init(const yplatform::ptree& xml) {
    auto pgg_info = xml.get_child("pgg");
    auto query_conf_file = pgg_info.get("query_conf", "");
    query_conf = pgg::readQueryConfFile(query_conf_file, query::QueriesRegister(), query::ParametersRegister());
    dbuser = pgg_info.get("dbuser", "");
    if (dbuser.empty()) {
        throw std::runtime_error("error: 'dbuser' is empty in the given config file (see 'pgg' section)");
    }

    sharpei_conf.init(xml.get_child("sharpei"));

    auto max_connections = pgg_info.get<std::size_t>("max_conn");
    pgg::Milliseconds connection_timeout(pgg_info.get<long long>("connection_timeout"));
    pgg::ConnectionPoolFactory poolFactory;
    poolFactory.maxConnections(max_connections)
        .connectTimeout(connection_timeout)
        .ioService(*yplatform::global_net_reactor->io());
    pool = poolFactory.product();
}

void impl::fini() {
}

pgg::SharpeiParams impl::create_sharpei_params(const TContextPtr& ctx) {
    pgg::SharpeiParams params;

    params.httpClient = std::make_shared<furita::SharpeiHttpLoader>(ctx, sharpei_conf.timeouts);
    params.settings = sharpei_conf.settings;
    return params;
}

pgg::RequestExecutor impl::create_request_executor(const TContextPtr& ctx,
                                             const uint64_t& id,
                                             const pgg::UidResolverFactoryPtr& resolverFactory,
                                             sharpei::client::Mode mode) {
    pgg::FactoryParams params;
    params.connPool = pool;
    params.queryConf = query_conf;
    params.requestInfo.requestId = ctx->uniq_id();
    params.logger = logging::getPggLogger(logging::makeContextLog(ctx));
    params.profiler = macs::pg::integration::PaProfiler::create();
    params.credentials = {dbuser, ""};
    if (mode == sharpei::client::Mode::WriteOnly) {
        return pgg::Factory::product(std::to_string(id), resolverFactory, params, pgg::MasterOnly());
    } else {
        return pgg::Factory::product(std::to_string(id), resolverFactory, params, pgg::ReadReplicaThenMaster());
    }
}

future<boost::unordered_map<uint64_t, RulesResult>>
impl::get_rules_multiuser(const TContextPtr& ctx,
                          const std::vector<uint64_t>& uids,
                          sharpei::client::Mode mode) {
    promise<boost::unordered_map<uint64_t, RulesResult>> prom;
    pgg::SharpeiParams params = create_sharpei_params(ctx);

    boost::shared_ptr<MultiuserRulesHandler> handler = boost::make_shared<MultiuserRulesHandler>(uids);
    sharpei::client::RequestInfo reqInfo{ctx->uniq_id(), "", "", "127.0.0.1", ""};
    const auto sharpeiClient = createSharpeiClient(params.httpClient, params.settings,
                                                   reqInfo);
    handler->GetUsersConnInfos(uids, sharpeiClient).get();
    auto shard_to_uids = handler->SplitUsersByShards();

    if (shard_to_uids.empty()) {
        prom.set(handler->users_with_rules_result);
    } else {
        for (auto& [ shard_id, shard_uids ] : shard_to_uids) {
            auto resolverFactory =  pgg::createSharpeiUidResolverFactory(params);
            auto executor = create_request_executor(ctx,
                                                    shard_uids[0],
                                                    resolverFactory,
                                                    mode);

            executor.request<std::vector<reflection::UnionRuleConditionAction>>(
                pgg::withQuery<query::GetAllRulesMultiuser>(query::Uids(shard_uids)),
                [=, uids=shard_uids](pgg::error_code ec, const std::vector<reflection::UnionRuleConditionAction>& res) mutable {
                    handler->GetRulesHandler(ec, res, prom, uids);
                });
        }
    }
    return prom;
}

future<rules::rule_list_ptr> impl::get_rule(pgg::RequestExecutor& executor,
                                            const uint64_t& uid, const uint64_t& rule_id) {
    promise<rules::rule_list_ptr> prom;
    boost::shared_ptr<GetRuleHandler> handler = boost::make_shared<GetRuleHandler>();

    executor.request<std::vector<reflection::Rule>>(
        pgg::withQuery<query::GetRule>(query::Uid(uid), query::RuleId(rule_id)),
        [=](pgg::error_code ec, const std::vector<reflection::Rule>& res) mutable {
            handler->HandleGetRule(ec, res, prom, rule_id);
        });
    return prom;
}

future<rules::rule_list_ptr> impl::get_all_rules(
        pgg::RequestExecutor& executor,
        const uint64_t& uid) {
    promise<rules::rule_list_ptr> prom;
    boost::shared_ptr<GetRulesHandler> handler = boost::make_shared<GetRulesHandler>();

    executor.request<std::vector<reflection::Rule>>(
        pgg::withQuery<query::GetAllRules>(query::Uid(uid)),
        [=](pgg::error_code ec, const std::vector<reflection::Rule>& res) mutable {
            handler->HandleGetRules(ec, res, prom);
        });
    return prom;
}

future<rules::rule_list_ptr> impl::get_rules_by_type(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const rules_helpers::RuleType& type) {
    promise<rules::rule_list_ptr> prom;
    boost::shared_ptr<GetRulesHandler> handler = boost::make_shared<GetRulesHandler>();

    executor.request<std::vector<reflection::Rule>>(
        pgg::withQuery<query::GetRulesByType>(query::Uid(uid), query::RuleType(type.toString())),
        [=](pgg::error_code ec, const std::vector<reflection::Rule>& res) mutable {
            handler->HandleGetRules(ec, res, prom);
        });
    return prom;
}

future<void> impl::remove_rules(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const std::vector<uint64_t>& ids) {
    promise<void> prom;

    executor.update(
        pgg::withQuery<query::RemoveRules>(query::Uid(uid), query::RulesIds(ids)),
        [=](pgg::error_code ec, int) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set();
            }
        });
    return prom;
}

future<void> impl::enable_rule(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const uint64_t& id, bool enabled) {
    promise<void> prom;

    if (enabled) {
        executor.update(
            pgg::withQuery<query::EnableRule>(query::Uid(uid), query::RuleId(id)),
            [=](pgg::error_code ec, int) mutable {
                if (ec) {
                    prom.set_exception(pgg::system_error(ec));
                } else {
                    prom.set();
                }
            });
    } else {
        executor.update(
            pgg::withQuery<query::DisableRule>(query::Uid(uid), query::RuleId(id)),
            [=](pgg::error_code ec, int) mutable {
                if (ec) {
                    prom.set_exception(pgg::system_error(ec));
                } else {
                    prom.set();
                }
            });
    }
    return prom;
}

future<rules::condition_list_ptr> impl::get_rule_conditions(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const uint64_t& rule_id) {
    promise<rules::condition_list_ptr> prom;
    boost::shared_ptr<GetConditionsHandler> handler = boost::make_shared<GetConditionsHandler>();

    executor.request<std::vector<reflection::Condition>>(
        pgg::withQuery<query::GetRuleConditions>(query::Uid(uid), query::RuleId(rule_id)),
        [=](pgg::error_code ec, const std::vector<reflection::Condition>& res) mutable {
            handler->HandleConditions(ec, res, prom);
        });
    return prom;
}

future<rules::condition_list_ptr> impl::get_all_conditions(
        pgg::RequestExecutor& executor,
        const uint64_t& uid) {
    promise<rules::condition_list_ptr> prom;
    boost::shared_ptr<GetConditionsHandler> handler = boost::make_shared<GetConditionsHandler>();

    executor.request<std::vector<reflection::Condition>>(
        pgg::withQuery<query::GetAllConditions>(query::Uid(uid)),
        [=](pgg::error_code ec, const std::vector<reflection::Condition>& res) mutable {
            handler->HandleConditions(ec, res, prom);
        });
    return prom;
}

future<rules::action_list_ptr> impl::get_rule_actions(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const uint64_t& rule_id) {
    promise<rules::action_list_ptr> prom;
    boost::shared_ptr<GetActionsHandler> handler = boost::make_shared<GetActionsHandler>();

    executor.request<std::vector<reflection::Action>>(
        pgg::withQuery<query::GetRuleActions>(query::Uid(uid), query::RuleId(rule_id)),
        [=](pgg::error_code ec, const std::vector<reflection::Action>& res) mutable {
            handler->HandleActions(ec, res, prom);
        });
    return prom;
}

future<rules::action_list_ptr> impl::get_all_actions(
        pgg::RequestExecutor& executor,
        const uint64_t& uid) {
    promise<rules::action_list_ptr> prom;
    boost::shared_ptr<GetActionsHandler> handler = boost::make_shared<GetActionsHandler>();

    executor.request<std::vector<reflection::Action>>(
        pgg::withQuery<query::GetAllActions>(query::Uid(uid)),
        [=](pgg::error_code ec, const std::vector<reflection::Action>& res) mutable {
            handler->HandleActions(ec, res, prom);
        });
    return prom;

}

future<rules::action_list_ptr> impl::get_action(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const uint64_t& action_id) {
    promise<rules::action_list_ptr> prom;
    boost::shared_ptr<GetActionsHandler> handler = boost::make_shared<GetActionsHandler>();

    executor.request<std::vector<reflection::Action>>(
        pgg::withQuery<query::GetAction>(query::Uid(uid), query::ActionId(action_id)),
        [=](pgg::error_code ec, const std::vector<reflection::Action>& res) mutable {
            handler->HandleActions(ec, res, prom);
        });
    return prom;
}

future<bool> impl::order_rule(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const std::vector<uint64_t>& ids) {
    promise<bool> prom;

    executor.request<reflection::RulesOrderResult>(
        pgg::withQuery<query::RuleOrderWithCheck>(query::Uid(uid), query::RulesIds(ids)),
        [=](pgg::error_code ec, const reflection::RulesOrderResult& res) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set(res.order_rules_with_check);
            }
        });
    return prom;
}

future<void> impl::verify_rule(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const uint64_t& id) {
    promise<void> prom;

    executor.update(
        pgg::withQuery<query::VerifyRule>(query::Uid(uid), query::ActionId(id)),
        [=](pgg::error_code ec, int) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set();
            }
        });
    return prom;
}

future<void> impl::edit_rule(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const rules::rule_ptr& rule, const boost::optional<uint64_t>& old_id, bool last) {
    promise<void> prom;
    boost::shared_ptr<EditRuleHandler> handler = boost::make_shared<EditRuleHandler>();

    executor.update(
        pgg::withQuery<query::RuleEdit>(query::Uid(uid), query::RuleName(rule->name), query::RuleEnabled(true),
                                        query::RuleStop(rule->stop), query::RuleLast(last),
                                        query::RuleActs(handler->PrepareActions(rule->actions)),
                                        query::RuleConds(handler->PrepareConditions(rule->conditions)),
                                        query::RuleOldId(old_id),
                                        query::RuleNewId(rule->id),
                                        query::RuleType(rule->type)),
        [=](pgg::error_code ec, int) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set();
            }
        });
    return prom;
}

future<std::map<uint64_t, std::pair<std::string, std::string>>> impl::get_not_verified_parameters(
        pgg::RequestExecutor& executor,
        const uint64_t& uid,
        const uint64_t& id) {
    promise<std::map<uint64_t, std::pair<std::string, std::string>>> prom;
    boost::shared_ptr<NotVerifiedParamsHandler> handler = boost::make_shared<NotVerifiedParamsHandler>();

    executor.request<std::vector<reflection::NotVerifiedParams>>(
        pgg::withQuery<query::GetNotVerifiedParams>(query::Uid(uid), query::RuleId(id)),
        [=](pgg::error_code ec, const std::vector<reflection::NotVerifiedParams>& res) mutable {
            handler->HandleNotVerifiedParams(ec, res, prom);
        });
    return prom;
}

future<uint64_t> impl::get_new_rule_id(
    pgg::RequestExecutor& executor,
    const uint64_t&) {
    promise<uint64_t> prom;

    executor.request<reflection::NewRuleId>(
        pgg::withQuery<query::GetNewRuleId>(),
        [=](pgg::error_code ec, const reflection::NewRuleId& res) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set(res.nextval);
            }
        });

    return prom;
}

future<rules::blacklist_ptr> impl::get_all_lists(
    pgg::RequestExecutor& executor,
    const uint64_t& uid) {
    promise<rules::blacklist_ptr> prom;
    boost::shared_ptr<BWListHandler> handler = boost::make_shared<BWListHandler>();

    executor.request<std::vector<reflection::ListEntry>>(
        pgg::withQuery<query::GetAllLists>(query::Uid(uid)),
        [=](pgg::error_code ec, const std::vector<reflection::ListEntry>& res) mutable {
            handler->HandleBWList(ec, res, prom);
        });
    return prom;
}

future<rules::blacklist_ptr> impl::get_list_by_type(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const std::string& type) {
    promise<rules::blacklist_ptr> prom;
    boost::shared_ptr<BWListHandler> handler = boost::make_shared<BWListHandler>();

    executor.request<std::vector<reflection::ListEntry>>(
        pgg::withQuery<query::GetListByType>(query::Uid(uid), query::ListType(type)),
        [=](pgg::error_code ec, const std::vector<reflection::ListEntry>& res) mutable {
            handler->HandleBWList(ec, res, prom);
        });
    return prom;
}

future<void> impl::remove_from_list(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const std::vector<std::string>& emails,
    const std::string& type) {
    promise<void> prom;

    executor.update(
        pgg::withQuery<query::RemoveFromList>(query::Uid(uid), query::ListType(type), query::Emails(emails)),
        [=](pgg::error_code ec, int) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set();
            }
        });
    return prom;
}

future<void> impl::add_to_list(
    pgg::RequestExecutor& executor,
    const uint64_t& uid,
    const std::string& email, const std::string& type) {
    promise<void> prom;

    executor.update(
        pgg::withQuery<query::AddToList>(query::Uid(uid), query::ListType(type), query::Email(email)),
        [=](pgg::error_code ec, int) mutable {
            if (ec) {
                prom.set_exception(pgg::system_error(ec));
            } else {
                prom.set();
            }
        });
    return prom;
}

}
}

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(furita::pq::impl)
