#include "impl.hpp"
#include "queries.hpp"
#include "furita/common/rule.hpp"

#include <sharpei_client/sharpei_client.h>
#include <pgg/factory.h>
#include <pgg/service/uid_resolver.h>

namespace furita {
namespace pq {
struct MultiuserRulesHandler : boost::enable_shared_from_this<MultiuserRulesHandler> {
    MultiuserRulesHandler(const std::vector<uint64_t>& uids)
        : users_count(uids.size())
        , uids(uids)
        , receive_conn_infos_count(0)
        , requests_by_rules_count(0)
    {
        for (const auto& uid : uids) {
            users_conn_infos.emplace(uid, UserConnInfoResult());
            users_with_rules_result.emplace(uid, RulesResult(boost::make_shared<rules::rule_list>()));
        }
    }

    future<void>
    GetUsersConnInfos(const std::vector<uint64_t>& uids, sharpei::client::SharpeiClientPtr sharpeiClient) {
        promise<void> prom;

        for (const auto& uid : uids) {
            sharpei::client::ResolveParams resolveParams(std::to_string(uid), sharpei::client::Mode::ReadWrite);
            sharpeiClient->asyncGetConnInfo(
                resolveParams,
                [ =, thiz = shared_from_this() ](const sharpei::client::ErrorCode& error, sharpei::client::Shard shard) mutable {
                    auto it = thiz->users_conn_infos.find(uid);
                    if (error) {
                        it->second = error;
                    } else {
                        it->second = shard.id;
                    }
                    if (++thiz->receive_conn_infos_count == thiz->users_count) {
                        prom.set();
                    }
                });
        }
        return prom;
    }

    boost::unordered_map<std::string, std::vector<uint64_t>>
    SplitUsersByShards() {
        boost::unordered_map<std::string, std::vector<uint64_t>> shard_uids;
        for (const auto & [ uid, usr_conn_info ] : users_conn_infos) {
            if (usr_conn_info.which() == 1) {
                auto it = users_with_rules_result.find(uid);
                it->second = boost::get<sharpei::client::ErrorCode>(usr_conn_info);
            } else {
                if (shard_uids.find(boost::get<std::string>(usr_conn_info)) != shard_uids.end()) {
                    shard_uids[boost::get<std::string>(usr_conn_info)].push_back(uid);
                } else {
                    shard_uids.emplace(boost::get<std::string>(usr_conn_info), std::vector<uint64_t>{uid});
                }
            }
        }

        requests_by_rules_count = shard_uids.size();
        return shard_uids;
    }

    rules::rule_ptr FillRule(const furita::pq::reflection::UnionRuleConditionAction& record) {
        rules::rule_ptr r = boost::make_shared<rules::rule>();
        r->uid = record.uid;
        r->id = record.rule_id;
        r->name = *record.name;
        r->enabled = *record.enabled;
        r->prio = *record.prio;
        r->stop = *record.stop;
        r->created = *record.created;
        r->type = *record.type;
        return r;
    }

    rules::condition_ptr FillCondition(const furita::pq::reflection::UnionRuleConditionAction& record) {
        rules::condition_ptr c = boost::make_shared<rules::condition>();
        c->rule_id = record.rule_id;
        c->field_type = *record.field_type;
        if (record.field) {
            c->field = *record.field;
        }
        if (record.pattern) {
            c->pattern = *record.pattern;
        }
        c->oper = rules::condition::oper_type::fromString(*record.condition_oper);
        c->link = rules::condition::link_type::fromString(*record.link);
        c->neg = *record.negative;
        return c;
    }

    rules::action_ptr FillAction(const furita::pq::reflection::UnionRuleConditionAction& record) {
        rules::action_ptr a = boost::make_shared<rules::action>();
        a->rule_id = record.rule_id;
        a->id = *record.action_id;
        a->oper = *record.action_oper;
        if (record.param) {
            a->param = *record.param;
        }
        a->verified = *record.verified;
        return a;
    }

    void FillUserWithRulesResult(const boost::unordered_map<uint64_t, rules::rule_list_ptr>& uid_with_rule_to_rules,
                                 const boost::unordered_map<std::pair<uint64_t, uint64_t>, rules::action_list_ptr>& uid_with_rule_to_actions,
                                 const boost::unordered_map<std::pair<uint64_t, uint64_t>, rules::condition_list_ptr>& uid_with_rule_to_conditions) {
        for (const auto& [ uid, rules_list ] : uid_with_rule_to_rules) {
            for (auto rule : *rules_list) {
                const auto uid_with_rule_id = std::make_pair(uid, rule->id);
                auto c_it = uid_with_rule_to_conditions.find(uid_with_rule_id);
                if (c_it != uid_with_rule_to_conditions.end()) {
                    rule->conditions = c_it->second;
                }
                auto a_it = uid_with_rule_to_actions.find(uid_with_rule_id);
                if (a_it != uid_with_rule_to_actions.end()) {
                    rule->actions = a_it->second;
                }
            }
            auto it = users_with_rules_result.find(uid);
            it->second = rules_list;
        }
    }

    void GetRulesHandler(pgg::error_code ec, std::vector<reflection::UnionRuleConditionAction> res,
                         promise<boost::unordered_map<uint64_t, RulesResult>> prom,
                         const std::vector<uint64_t>& current_uids) {
        if (ec) {
            for (auto uid : current_uids) {
                auto it = users_with_rules_result.find(uid);
                it->second = ec;
            }
        } else {
            boost::unordered_map<uint64_t, rules::rule_list_ptr> uid_with_rule_to_rules;
            boost::unordered_map<std::pair<uint64_t, uint64_t>, rules::action_list_ptr> uid_with_rule_to_actions;
            boost::unordered_map<std::pair<uint64_t, uint64_t>, rules::condition_list_ptr> uid_with_rule_to_conditions;
            for (const auto& elem : res) {
                const uint64_t uid = elem.uid, rule_id = elem.rule_id;
                const auto uid_with_rule = std::make_pair(uid, rule_id);

                if (elem.name) {
                    const auto r = FillRule(elem);
                    if (uid_with_rule_to_rules.find(uid) != uid_with_rule_to_rules.end()) {
                        uid_with_rule_to_rules[uid]->push_back(r);
                    } else {
                        rules::rule_list_ptr rule_list(new rules::rule_list());
                        rule_list->push_back(r);
                        uid_with_rule_to_rules.emplace(uid, rule_list);
                    }
                } else if (elem.field_type) {
                    const auto c = FillCondition(elem);
                    if (uid_with_rule_to_conditions.find(uid_with_rule) != uid_with_rule_to_conditions.end()) {
                        uid_with_rule_to_conditions[uid_with_rule]->push_back(c);
                    } else {
                        rules::condition_list_ptr condition_list(new rules::condition_list());
                        condition_list->push_back(c);
                        uid_with_rule_to_conditions.emplace(uid_with_rule, condition_list);
                    }
                } else if (elem.action_id) {
                    const auto a = FillAction(elem);
                    if (uid_with_rule_to_actions.find(uid_with_rule) != uid_with_rule_to_actions.end()) {
                        uid_with_rule_to_actions[uid_with_rule]->push_back(a);
                    } else {
                        rules::action_list_ptr action_list(new rules::action_list());
                        action_list->push_back(a);
                        uid_with_rule_to_actions.emplace(uid_with_rule, action_list);
                    }
                }
            }
            FillUserWithRulesResult(uid_with_rule_to_rules, uid_with_rule_to_actions, uid_with_rule_to_conditions);
        }
        if (--requests_by_rules_count == 0) {
            prom.set(std::move(users_with_rules_result));
        }
    }

    size_t users_count;
    std::vector<uint64_t> uids;
    std::atomic<size_t> receive_conn_infos_count;
    std::atomic<size_t> requests_by_rules_count;
    boost::unordered_map<uint64_t, UserConnInfoResult> users_conn_infos;
    boost::unordered_map<uint64_t, RulesResult> users_with_rules_result;
};

}
}
