#include "impl.h"
#include "msearch_call.hpp"
#include "mops_call.hpp"
#include "make_search_helper.hpp"

#include <furita/common/context.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>
#include <yplatform/find.h>
#include <yplatform/exception.h>
#include <ymod_blackbox/auth.h>
#include <ymod_tvm/module.h>

#include <memory>

namespace furita::processor {

namespace {

template <typename Future>
std::string exception_description(Future& f)
{
    if (!f.has_exception())
        return "no exceptions";
    try { f.get(); }
    catch (const yplatform::exception& e) { return e.public_message(); }
    catch (const std::exception& e) { return e.what(); }
    catch (...) {}
    return "unknown";
}

} // namespace

struct impl::ApplyHandler : public std::enable_shared_from_this<ApplyHandler> {
    ApplyHandler(
        const TContextPtr &context,
        const uint64_t& uid,
        const std::string& remote_ip,
        const std::string& tvm,
        const ImplPtr &proc,
        boost::shared_ptr<configuration> config,
        const ymod_httpclient::headers_dict& headers
    )
        : m_context(context)
        , httpClient_(new HttpClientImpl)
        , headers(headers)
        , uid_(uid)
        , remote_ip_(remote_ip)
        , tvm_(tvm)
        , m_search_offset(0)
        , m_retry_count(0)
        , m_delete(false)
        , proc_(proc)
        , config_(config)
    {}

    void run(const uint64_t& id, bool master) {
        m_id = id;

        mode = master
                   ? sharpei::client::Mode::WriteOnly
                   : sharpei::client::Mode::ReadWrite;

        auto pq = yplatform::find<furita::pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(m_context));
        auto executor = pq->create_request_executor(m_context, uid_, resolverFactory, mode);

        auto futureRule = pq->get_rule(executor, uid_, m_id);
        auto futureRuleActions = pq->get_rule_actions(executor, uid_, m_id);

        futureRule.add_callback(
            std::bind(&ApplyHandler::handleRule, shared_from_this(), executor, futureRule, futureRuleActions)
        );
    }

    promise<void> promise;

private:
    void handleRule(
        pgg::RequestExecutor& executor,
        future<rules::rule_list_ptr> result,
        future<rules::action_list_ptr> actions_result)
    {
        if (result.has_exception() || result.get()->size() != 1) {
            promise.set_exception(yplatform::exception("Error", "Failed to get rule"));
            return;
        }
        rule = result.get()->front();
        actions_result.add_callback(
            std::bind(&ApplyHandler::handleActions, shared_from_this(), executor, actions_result)
        );
    }

    void handleActions(pgg::RequestExecutor& executor, future<rules::action_list_ptr> actions_result) {
        if (actions_result.has_exception()) {
            promise.set_exception(yplatform::exception("Error", "Failed to get rule actions"));
            return;
        }
        rule->actions = actions_result.get();

        for(const rules::action_ptr &a : *(rule->actions)) {
            if (m_folder.empty() && a->oper == "move") {
                m_folder = a->param;
            } else if (m_folder.empty() && a->oper == "delete") {
                m_delete = true;
            } else if (m_status.empty() && a->oper == "status") {
                m_status = a->param;
            } else if (a->oper == "movel") {
                m_labels.push_back(a->param);
            }
        }

        if (m_folder.empty() && m_status.empty() && m_labels.empty() && !m_delete) {
            return handle_error("Failed to apply rule actions: not supported");
        }

        auto pq = yplatform::find<furita::pq::pq>("furita_pq");
        auto futureConditions = pq->get_rule_conditions(executor, uid_, m_id);
        futureConditions.add_callback(
            std::bind(&ApplyHandler::handleConditions, shared_from_this(), futureConditions)
        );
    }


    void handleConditions(future<rules::condition_list_ptr> result) {
        if (result.has_exception()) {
            promise.set_exception(yplatform::exception("Error", "Failed to get rule conditions"));
            return;
        }

        rule->conditions = result.get();

        try {
            m_search_query = msq::make_search_query(rule, std::to_string(uid_));
        } catch (const std::exception &e) {
            return promise.set_exception(yplatform::exception("Error", e.what()));
        }

        get_messages();
    }

    void get_messages() {
        messages.clear();
        const auto& opts = config_->searchOptions;
        const auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
        auto call = std::make_shared<MsearchCall>(m_context, httpClient_, opts.fetchUrl, headers,
            opts.timeouts, tvm_, tvmModule);
        auto futureMids = call->search(
            "",
            std::to_string(uid_),
            m_search_query,
            m_search_offset,
            opts.fetchLimit,
            remote_ip_,
            opts.folderSet,
            config_->log_pa
        );
        futureMids.add_callback(
            std::bind(&ApplyHandler::handle_search, shared_from_this(), futureMids, call)
        );
    }

    void handle_search(future<void> futureMids, MsearchCallPtr call) {
        if (futureMids.has_exception()) {
            if (m_retry_count++ < config_->searchOptions.attempts) {
                return get_messages();
            } else {
                return handle_error("Failed to get messages");
            }
        }
        m_retry_count = 0;

        messages.swap(call->mids());

        FURITA_LOG_DEBUG(m_context, logdog::message="message list: count=" + std::to_string(messages.size()))

        if (messages.empty()) {
            FURITA_LOG_NOTICE(m_context, logdog::message="apply operation finished: status=ok")
            return promise.set();
        }

        m_search_offset += static_cast<unsigned int>(messages.size());

        apply_move();
    }

    void apply_move() {
        const auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
        if (m_delete) {
            future<void> f = boost::make_shared<mops_call>(m_context, httpClient_, tvm_, headers, tvmModule)->
                remove(config_->mops, uid_, messages);
            f.add_callback(std::bind(&ApplyHandler::handle_move, shared_from_this(), f));
        } else if (!m_folder.empty()) {
            future<void> f = boost::make_shared<mops_call>(m_context, httpClient_, tvm_, headers, tvmModule)->
                move(config_->mops, uid_, messages, m_folder);
            f.add_callback(std::bind(&ApplyHandler::handle_move, shared_from_this(), f));
        } else {
            apply_label();
        }
    }

    void handle_move(future<void> result) {
        if (result.has_exception()) {
            return handle_error("Failed to apply move action");
        }
        FURITA_LOG_NOTICE(m_context, logdog::message="rule action planned: type=move, folder=" + m_folder)
        apply_label();
    }

    void apply_label() {
        const auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
        if (!m_labels.empty()) {
            future<void> f =
                boost::make_shared<mops_call>(m_context, httpClient_, tvm_, headers, tvmModule)->
                label(config_->mops, uid_, messages, m_labels);
            f.add_callback(std::bind(&ApplyHandler::handle_label, shared_from_this(), f));
        } else {
            apply_status();
        }
    }

    void handle_label(future<void> result) {
        if (result.has_exception()) {
            return handle_error("Failed to apply label action");
        }
        FURITA_LOG_NOTICE(m_context, logdog::message="rule action planned: type=label")
        apply_status();
    }

    void apply_status() {
        const auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
        if (!m_status.empty()) {
            future<void> f =
                boost::make_shared<mops_call>(m_context, httpClient_, tvm_, headers, tvmModule)->
                mark(config_->mops, uid_, messages, m_status);
            f.add_callback(std::bind(&ApplyHandler::handle_status,
                shared_from_this(), f));
        } else {
            handle_apply();
        }
    }

    void handle_status(future<void> result) {
        if (result.has_exception()) {
            return handle_error("Failed to apply status action");
        }
        FURITA_LOG_NOTICE(m_context, logdog::message="rule action planned: type=update, status=" + m_status)
        handle_apply();
    }

    void handle_apply() {
        if (messages.size() < config_->searchOptions.fetchLimit) {
            FURITA_LOG_NOTICE(m_context, logdog::message="apply operation finished: status=ok")
            return promise.set();
        }
        get_messages();
    }

    void handle_error(const std::string &message) {
        FURITA_LOG_ERROR(m_context, logdog::message="apply operation finished: status=error, report='" + message + "'")
        promise.set_exception(yplatform::exception("Error", message));
    }

    TContextPtr m_context;
    HttpClientPtr httpClient_;
    ymod_httpclient::headers_dict headers;

    uint64_t uid_, m_id;
    std::string remote_ip_;
    std::string tvm_;
    sharpei::client::Mode mode;

    unsigned int m_search_offset;
    std::string m_search_query;

    std::vector<std::string> messages;

    unsigned int m_retry_count;

    std::string m_folder, m_status;
    std::vector<std::string> m_labels;
    bool m_delete;

    rules::rule_ptr rule;

    ImplPtr proc_;
    boost::shared_ptr<configuration> config_;
};

future<void> impl::apply(
    const TContextPtr &context,
    const uint64_t& uid,
    const uint64_t& id,
    bool master,
    const std::string& remote_ip,
    const std::string& tvm,
    const ymod_httpclient::headers_dict& headers)
{
    auto handler = std::make_shared<ApplyHandler>(
        context,
        uid,
        remote_ip,
        tvm,
        std::static_pointer_cast<impl>(shared_from_this()),
        m_configuration,
        headers
    );
    handler->run(id, master);
    return handler->promise;
}

}   // namespace furita::processor
