#include <lib/idna.h>
#include <yplatform/find.h>
#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/encoding/url_encode.h>
#include <ymod_smtpclient/call.h>
#include <ymod_httpclient/call.h>
#include <furita/common/context.h>
#include <furita/pq/pq.hpp>

#include <boost/format.hpp>
#include "impl.h"
#include "verification_token.h"

using namespace furita;
using namespace furita::processor;


template <typename Stream>
void expand_tokens(const std::string &source, const std::map<std::string, std::string> &tokens, Stream &result)
{
    for (std::string::const_iterator begin = source.begin(), end = source.end(); begin != end;)
    {
        if (*begin == '$')
        {
            // get token name
            std::string::const_iterator i = ++begin;
            for (; i != end && (std::isalnum(*i) || *i == '_'); ++i);
            std::string name(begin, i);

            // get token value
            std::map<std::string, std::string>::const_iterator value = tokens.find(name);
            if (value == tokens.end())
                result << "$" << name;
            else
                result << value->second;
            begin = i;
        }
        else
            result << *begin++;
    }
}

namespace
{
    struct transform_handler : public ymod_http_client::response_handler
    {
        void handle_data(const char *data, unsigned long long sz) override
        {
            data_.append(data, sz);
        }

        std::string data_;
    };

    void handle_transform(ymod_http_client::future_void_t result,
        boost::shared_ptr<transform_handler> handler, promise<std::string> prom,
        const std::string& default_param)
    {
        if (result.has_exception())
            prom.set(default_param);
        else
            prom.set(handler->data_);
    }

    future<std::string> transform_url(const TContextPtr &ctx,
        const std::string& need_to_transform, const std::string& host)
    {
        promise<std::string> prom;
        auto handler = boost::shared_ptr<transform_handler>();

        auto httpClient = yplatform::find<yhttp::call>("http_client");
        auto rmInfo = httpClient->make_rm_info(host);

        std::ostringstream request;
        request << "?url=" << need_to_transform;

        auto futureUrl = httpClient->get_url(ctx->CreateTaskContext(), handler, rmInfo, request.str());
        futureUrl.add_callback(boost::bind(
            handle_transform, futureUrl, handler, prom, need_to_transform));

        return prom;
    }
}

struct edit_data : public boost::enable_shared_from_this<edit_data>
{
    explicit edit_data(const configuration_ptr &configuration,
        const TContextPtr &context,
        const uint64_t& uid,
        const ImplPtr &proc) :
        m_configuration(configuration), m_context(context), /*m_user(user),*/
        uid_(uid), m_confirm(false), proc_(proc)
    {}

    void set_confirm_parameters(const std::string &domain, const std::string &auth_domain, const std::string &from, const std::string &lang)
    {
        if (from.empty() || domain.empty() || auth_domain.empty() || lang.empty())
            return;

        m_from = from;
        m_domain = domain;
        m_auth_domain = auth_domain;
        m_lang = lang;
        m_confirm = true;
        action_confirm_.clear();
    }

    void run(const boost::optional<uint64_t>& id, const rules::rule_ptr& rule, bool last)
    {
        bool is_forward_rule = false;
        for (auto a : *rule->actions) {
            if (a->oper == "forward" ||
                a->oper == "forwardwithstore"){
                    is_forward_rule = true;
                    break;
            }
        }

        if (is_forward_rule && m_configuration->allow_forward == "none") {
            promise.set(
                std::make_pair(
                    status_result::FAIL,
                    boost::optional<uint64_t>()
                )
            );
        } else {
            m_order = last;
            m_old_id = id;
            rule_ = rule;

            if (m_confirm) {
                for (const rules::action_ptr &a : *rule_->actions) {
                    if (a->oper == "forward" ||
                        a->oper == "forwardwithstore" ||
                        a->oper == "notify")
                    {
                        action_confirm_ = a->oper;
                        a->verified = false;
                    }
                }
            }

            auto pq = yplatform::find<pq::pq>("furita_pq");
            auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(m_context));
            auto executor = pq->create_request_executor(m_context, uid_, resolverFactory);

            future<uint64_t> r = pq->get_new_rule_id(executor, uid_);
            r.add_callback(boost::bind(&edit_data::handle_new_id, shared_from_this(), executor, r));
        }
    }

    promise<std::pair<status_result, boost::optional<uint64_t>>> promise;

private:
    void handle_new_id(pgg::RequestExecutor& executor, future<uint64_t> result) {
        if (result.has_exception())
            return promise.set_exception(yplatform::exception("Error", "Failed to get new rule id"));

        rule_->id = result.get();

        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto r = pq->edit_rule(executor, uid_, rule_, m_old_id, m_order);
        r.add_callback(boost::bind(&edit_data::handle_add, shared_from_this(), executor, r));
    }

    void handle_add(pgg::RequestExecutor& executor, future<void> result) {
        if (result.has_exception()) {
            return promise.set_exception(yplatform::exception("Error", "Failed to create new rule"));
        }

        if (m_confirm) {
            auto pq = yplatform::find<pq::pq>("furita_pq");
            auto r = pq->get_not_verified_parameters(
                executor, uid_, rule_->id);
            r.add_callback(boost::bind(&edit_data::handle_unverified_parameters,
                shared_from_this(), r));
        } else {
            promise.set(
                std::make_pair(
                    status_result::SUCCESS,
                    rule_->id
                )
            );
        }
    }

    void handle_unverified_parameters(future<std::map<uint64_t, std::pair<std::string, std::string>>> result)
    {
        if (result.has_exception() || result.get().empty()) {
            return promise.set(
                std::make_pair(
                    status_result::SUCCESS,
                    rule_->id
                )
            );
        }

        result.get().swap(m_unverified_parameters);
        check_parameters();
    }

    void check_parameters()
    {
        if (m_unverified_parameters.empty()) {
             return promise.set(
                std::make_pair(
                    status_result::SUCCESS,
                    rule_->id
                )
            );
        }

        auto p = m_unverified_parameters.begin();
        //p : {action_id, {not_verified_address, action_name}}
        action_confirm_ = p->second.second;

        if (m_configuration->filter_confirm_templates.empty())
            return promise.set_exception(yplatform::exception("Error", "Failed to send confirm request: no templates"));

        auto message_template =
                m_configuration->filter_confirm_templates.find(action_confirm_, m_lang, "forward", "ru");
        send_request(p->second.first, p->first, message_template);
        m_unverified_parameters.erase(p);
    }

    struct message_parameters
    {
        std::map<std::string, std::string> tokens;
        std::string rcpt;
        std::string from;
        std::string message_template;
    };
    typedef boost::shared_ptr<message_parameters> message_parameters_ptr;

    void send_request(const std::string &email, const uint64_t &id, const std::string &message_template)
    {
        if (message_template.empty()) {
            return promise.set_exception(
                    yplatform::exception("Error", "Failed to send confirm request: empty template"));
        }

        const std::string envelope_from("devnull@" + m_auth_domain);

        const std::string rnd = yplatform::url_encode<std::string>(make_token(id, uid_, m_from));
        const std::string link = (boost::format(
            "https://%1%/forward-confirm?e=%2%") % m_domain % rnd).str();

        message_parameters_ptr params(new message_parameters);
        params->tokens["link"] = link;
        params->tokens["raddr"] = email;
        params->tokens["from_a"] = m_from;
        params->tokens["envelopeFrom"] = envelope_from;

        if (!m_configuration->shortener_url.empty()) {
            params->from = m_from;
            params->rcpt = email;
            params->message_template = message_template;

            future<std::string> r = transform_url(m_context, link,
                m_configuration->shortener_url);
            r.add_callback(boost::bind(&edit_data::handle_transform_url,
                shared_from_this(), r, params));
        } else {
            std::stringstream message_stream;
            expand_tokens(message_template, params->tokens, message_stream);
            message_stream << "\r\n.\r\n";

            const auto smtpClient = yplatform::find<ymod_smtpclient::Call>("smtp_client");
            const auto callback = boost::bind(&edit_data::handle_send_request, shared_from_this(),
                _1, _2);
            ymod_smtpclient::RequestBuilder requestBuilder;
            requestBuilder.address(m_configuration->smtpAddr)
                .mailfrom(envelope_from).addRcpt(email).message(message_stream.str());
            smtpClient->asyncRun(m_context->CreateTaskContext(), requestBuilder.release(), callback);
        }
    }

    void handle_send_request(ymod_smtpclient::error::Code e, ymod_smtpclient::Response)
    {
        if (e) {
            return promise.set_exception(yplatform::exception("Error",
                std::string("Failed to send confirm request: ") + ymod_smtpclient::error::message(e)));
        }

        check_parameters();
    }

    void handle_transform_url(future<std::string> link, message_parameters_ptr params)
    {
        if (!link.has_exception())
            params->tokens["link"] = link.get();

        std::stringstream message_stream;
        expand_tokens(params->message_template, params->tokens, message_stream);
        message_stream << "\r\n.\r\n";

        const auto smtpClient = yplatform::find<ymod_smtpclient::Call>("smtp_client");
        const auto callback = boost::bind(&edit_data::handle_send_request, shared_from_this(),
            _1, _2);
        ymod_smtpclient::RequestBuilder requestBuilder;
        requestBuilder.address(m_configuration->smtpAddr)
            .mailfrom(params->from).addRcpt(params->rcpt).message(message_stream.str());
        smtpClient->asyncRun(m_context->CreateTaskContext(), requestBuilder.release(), callback);
    }

    configuration_ptr m_configuration;

    TContextPtr m_context;

    uint64_t uid_;
    std::string m_from, m_domain, m_auth_domain, m_lang;
    bool m_confirm;
    std::string action_confirm_;

    boost::optional<uint64_t> m_old_id;
    unsigned int m_order;
    std::map<uint64_t, std::pair<std::string, std::string>> m_unverified_parameters;
    rules::rule_ptr rule_;

    ImplPtr proc_;
};

future<std::pair<status_result, boost::optional<uint64_t>>> impl::rule_edit(const TContextPtr &ctx,
    const uint64_t& uid,
    const boost::optional<uint64_t>& id, const rules::rule_ptr &rule, bool last,
    const std::string &from, const std::string &lang,
    const std::string &confirm_domain, const std::string &auth_domain)
{
    boost::shared_ptr<edit_data> data(new edit_data(m_configuration, ctx, /*user,*/
        uid, std::static_pointer_cast<impl>(shared_from_this())));
    data->set_confirm_parameters(confirm_domain, auth_domain, from, lang);
    data->run(id, rule, last);
    return data->promise;
}

