#include <boost/algorithm/string/case_conv.hpp>
#include <boost/format.hpp>
#include <yplatform/find.h>
#include <yplatform/exception.h>
#include <ymod_blackbox/auth.h>
#include <furita/pq/pq.hpp>
#include <furita/common/context.h>
#include <furita/common/logger.h>
#include "impl.h"
#include "msearch_call.hpp"
#include "make_search_helper.hpp"

namespace furita {
namespace processor {

namespace
{
    template <typename Future>
    std::string exception_description(Future& f)
    {
        if (!f.has_exception())
            return "no exceptions";
        try { f.get(); }
        catch (const yplatform::exception& e) { return e.public_message(); }
        catch (const std::exception& e) { return e.what(); }
        catch (...) {}
        return "unknown";
    }
}

struct impl::PreviewHandler: public std::enable_shared_from_this<PreviewHandler> {
    explicit PreviewHandler(
        const TContextPtr &context,
        const uint64_t& uid,
        const std::string& remoteIp,
        const std::string& tvm,
        const ImplPtr &proc,
        const ymod_httpclient::headers_dict& headers
    )
        : ctx(context)
        , httpClient(new HttpClientImpl)
        , headers(headers)
        , uid(uid)
        , remoteIp(remoteIp)
        , tvm(tvm)
        , processorImpl(proc)
    {}

    void run(const uint64_t& id, unsigned int length, unsigned int offset) {
        ruleId = id;
        requiredCount = length;
        requiredOffset = offset;

        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, uid, resolverFactory);

        auto futureRules = pq->get_rule(executor, uid, ruleId);
        futureRules.add_callback(
            boost::bind(&PreviewHandler::handleRule, shared_from_this(), executor, futureRules)
        );
    }

    void run(const rules::rule_ptr &rule, unsigned int count, unsigned int offset) {
        requiredCount = count;
        requiredOffset = offset;

        if (rule->conditions->empty()) {
            promise.set_exception(yplatform::exception("Error", "No conditions"));
            return;
        }
        run(rule);
    }

    promise<std::vector<std::string>> promise;

private:
    using FutureBBResponse = yplatform::future::future<ymod_blackbox::response>;

    void run(const rules::rule_ptr &rule) {
        try {
            searchQuery = msq::make_search_query(rule, std::to_string(uid));
        } catch (const std::exception &e) {
            promise.set_exception(yplatform::exception("Error", e.what()));
            return;
        }
        getMessages();
    }

    void handleRule(pgg::RequestExecutor& executor, future<rules::rule_list_ptr> result) {
        if (result.has_exception() || result.get()->size() != 1) {
            promise.set_exception(yplatform::exception("Error", "Failed to get rule"));
            return;
        }

        rule = result.get()->front();

        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto futureConditions = pq->get_rule_conditions(executor, uid, ruleId);
        futureConditions.add_callback(
            boost::bind(&PreviewHandler::handleConditions, shared_from_this(), futureConditions)
        );
    }

    void handleConditions(future<rules::condition_list_ptr> result) {
        if (result.has_exception()) {
            promise.set_exception(yplatform::exception("Error", "Failed to get rule conditions"));
            return;
        }

        rule->conditions = result.get();

        run(rule);
    }

    void getMessages() {
        const auto& opts = processorImpl->m_configuration->searchOptions;
        const auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");
        auto call = std::make_shared<MsearchCall>(ctx, httpClient, opts.fetchUrl, headers, opts.timeouts, tvm, tvmModule);
        messages.clear();
        auto futureMids = call->search(
            "",
            std::to_string(uid),
            searchQuery,
            requiredOffset,
            requiredCount,
            remoteIp,
            opts.folderSet,
            processorImpl->m_configuration->log_pa
        );
        futureMids.add_callback(
            boost::bind(&PreviewHandler::handleSearch, shared_from_this(), futureMids, call)
        );
    }

    void handleSearch(future<void> futureMids, MsearchCallPtr call) {
        if (futureMids.has_exception()) {
            if (retryCount++ < processorImpl->m_configuration->searchOptions.attempts) {
                getMessages();
            } else {
                promise.set_exception(yplatform::exception("Error", "Failed to get messages"));
            }
            return;
        }
        retryCount = 0;

        messages.swap(call->mids());
        FURITA_LOG_DEBUG(ctx, logdog::message="message list: count=" + std::to_string(messages.size()))
        promise.set(messages);
    }

    TContextPtr ctx;
    HttpClientPtr httpClient;
    ymod_httpclient::headers_dict headers;

    rules::rule_ptr rule;

    uint64_t uid;
    std::string remoteIp;
    const std::string tvm;

    ImplPtr processorImpl;

    std::vector<std::string> messages;

    std::string searchQuery;
    uint64_t ruleId;

    std::size_t requiredOffset = 0;
    std::size_t requiredCount = 10;

    unsigned int retryCount = 0;
};

future<std::vector<std::string>> impl::preview(const TContextPtr &context,
    const uint64_t& uid,
    const uint64_t& id, unsigned int length, unsigned int offset,
    const std::string& remote_ip, const std::string& tvm,
    const ymod_httpclient::headers_dict& headers)
{
    auto handler = std::make_shared<PreviewHandler>(
        context, uid, remote_ip, tvm, std::static_pointer_cast<impl>(shared_from_this()), headers
    );
    handler->run(id, length, offset);
    return handler->promise;
}

future<std::vector<std::string>> impl::preview(const TContextPtr &context,
    const uint64_t& uid,
    const rules::rule_ptr &rule, unsigned int length,
    unsigned int offset, const std::string& remote_ip, const std::string& tvm,
    const ymod_httpclient::headers_dict& headers)
{
    auto handler = std::make_shared<PreviewHandler>(
        context, uid, remote_ip, tvm, std::static_pointer_cast<impl>(shared_from_this()), headers
    );
    handler->run(rule, length, offset);
    return handler->promise;
}

}   // namespace processor
}   // namespace furita
