#include "mops_call.hpp"
#include "http_client.hpp"
#include <boost/bind.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <contrib/libs/yajl/api/yajl_parse.h>
#include <yplatform/find.h>
#include <ymod_httpclient/call.h>
#include <pa/async.h>
#include <furita/common/context.h>
#include <furita/common/http_headers.h>
#include <furita/common/logger.h>

namespace furita {

typedef yplatform::future::promise<void> promise_void;

mops_call::mops_call(TContextPtr ctx, HttpClientPtr httpClient,
        const std::string& tvm, const ymod_httpclient::headers_dict& headers,
        TTvmModulePtr tvmModule)
    : ctx_(ctx)
    , httpClient_(httpClient)
    , tvm_(tvm)
    , headers_(headers)
    , tvmModule_(std::move(tvmModule))
{}

class mops_response_handler : public ymod_http_client::response_handler
{
public:
    mops_response_handler() : code_(0), good_(true), state_(state_none)
    {
        parser_ = yajl_alloc(&cbs_, NULL, this);
    }

    ~mops_response_handler()
    {
        yajl_free(parser_);
    }

    ymod_http_client::handler_version version() const override
    {
        return ymod_http_client::handler_version_already_parse_body;
    }

    void set_code(int code, const std::string &) override
    {
        code_ = code;
    }

    void handle_data(const char* data, unsigned long long size) override
    {
        if (good_)
        {
            int status = yajl_parse(parser_,
                reinterpret_cast<const unsigned char*>(data), size);
            good_ = (status == yajl_status_ok);
        }
    }

    void handle_data_end() override
    {
        if (good_)
            good_ = (yajl_complete_parse(parser_) == yajl_status_ok);
    }

    int code() const override
    {
        return code_;
    }

    const std::string& error_message() const
    {
        return error_msg_;
    }

private:
    static int handle_key(void* ctx, const unsigned char* str, std::size_t sz)
    {
        const char* s = reinterpret_cast<const char*>(str);
        if (strncasecmp(s, "error", sz) == 0)
            static_cast<mops_response_handler*>(ctx)->state_ = state_error;
        else
            static_cast<mops_response_handler*>(ctx)->state_ = state_none;
        return 1;
    }

    static int handle_str(void* ctx, const unsigned char* str, std::size_t sz)
    {
        mops_response_handler* h =
            static_cast<mops_response_handler*>(ctx);
        if (h->state_ == state_error)
        {
            h->error_msg_.assign(reinterpret_cast<const char*>(str), sz);
        }
        return 1;
    }

    int code_;
    std::string error_msg_;
    bool good_;
    yajl_handle parser_;
    static const yajl_callbacks cbs_;

    enum state_t
    {
        state_none = 0, state_error
    } state_;
};

const yajl_callbacks mops_response_handler::cbs_ = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    &mops_response_handler::handle_str,
    NULL,
    &mops_response_handler::handle_key,
    NULL,
    NULL,
    NULL
};

namespace {

template <typename Future>
std::string exception_description(Future& f)
{
    if (!f.has_exception())
        return "no exceptions";
    try { f.get(); }
    catch (const yplatform::exception& e) { return e.public_message(); }
    catch (const std::exception& e) { return e.what(); }
    catch (...) {}
    return "unknown";
}

class request_op : public boost::enable_shared_from_this<request_op>
{
public:
    request_op(TContextPtr ctx,
            HttpClientPtr httpClient,
            const mops_call_options& opt,
            const std::string& url,
            ymod_http_client::string_ptr data,
            const std::string& tvm,
            const ymod_httpclient::headers_dict& headers,
            TTvmModulePtr tvmModule) :
        ctx_(ctx),
        httpClient_(httpClient),
        opt_(opt),
        url_(url),
        data_(data),
        try_(0),
        tvm_(tvm),
        headers_(headers),
        tvmModule_(std::move(tvmModule))
    {
        timeouts_.connect = opt.connectTimeout;
        timeouts_.total = opt.requestTimeout;
    }

    future<void> run()
    {
        post_request();
        return prom_;
    }

private:
    void post_request()
    {
        if (++try_ > opt_.attempts * 2)
        {
            prom_.set_exception(yplatform::exception(
                "Error", "Failed to plan operation"));
            return;
        }

        const std::string& host = (try_ > opt_.attempts)
            ? opt_.fallback_host : opt_.primary_host;

        if (host.empty())
        {
            prom_.set_exception(yplatform::exception(
                "Error", "Failed to plan operation"));
            return;
        }

        std::string serviceTicket;
        const auto errorCode = tvmModule_->get_service_ticket("mops", serviceTicket);
        if (errorCode) {
            FURITA_LOG_ERROR(ctx_, logdog::message="tvm error", logdog::error_code=errorCode);
            prom_.set_exception(yplatform::exception("Error", "failed to obtain tvm ticket"));
            return;
        }

        auto handler = boost::make_shared<mops_response_handler>();
        auto futureResponse = httpClient_->post(ctx_, host, timeouts_, url_, data_,
            make_headers(std::move(serviceTicket)), handler);
        futureResponse.add_callback(boost::bind(
            &request_op::handle_response, shared_from_this(), futureResponse, handler));
    }

    void handle_response(ymod_http_client::future_void_t f,
        boost::shared_ptr<mops_response_handler> handler)
    {
        using namespace std::string_literals;

        if (f.has_exception()) {
            FURITA_LOG_ERROR(ctx_, logdog::message="mops: try="s + std::to_string(try_) + ", status=error (" + exception_description(f) + ")")
            post_request();
            return;
        }

        const auto code = handler->code();
        if (code != 200) {
            FURITA_LOG_ERROR(ctx_, logdog::message="mops: try="s + std::to_string(try_) + ", status=error (" + handler->error_message() + ")")
            post_request();
            return;
        }

        FURITA_LOG_NOTICE(ctx_, logdog::message="mops: try="s + std::to_string(try_) + ", status=ok")
        prom_.set();
    }

    ymod_httpclient::headers_dict make_headers(std::string serviceTicket) const {
        ymod_httpclient::headers_dict res = headers_;
        if (!tvm_.empty()) {
            res.emplace(HttpHeaderNames::ticket, tvm_);
        }
        res.emplace(HttpHeaderNames::tvmServiceTicket, std::move(serviceTicket));
        return res;
    }

    TContextPtr ctx_;
    HttpClientPtr httpClient_;
    const mops_call_options& opt_;
    std::string url_;
    ymod_http_client::string_ptr data_;
    ymod_http_client::timeouts timeouts_;
    unsigned short try_;
    promise_void prom_;
    const std::string tvm_;
    const ymod_httpclient::headers_dict headers_;
    TTvmModulePtr tvmModule_;
};

future<void> post_request(TContextPtr ctx,
    HttpClientPtr httpClient,
    const mops_call_options& opt, const std::string& url,
    ymod_http_client::string_ptr data,
    const std::string& tvm,
    const ymod_httpclient::headers_dict& headers,
    TTvmModulePtr tvmModule)
{
    return boost::make_shared<request_op>(ctx, httpClient, opt, url, data, tvm, headers, std::move(tvmModule))->run();
}

template <typename Iterator>
inline void fill_field(const ymod_http_client::string_ptr& data,
    const char* name, Iterator begin, Iterator end)
{
    if (begin != end)
    {
        data->append(1, '&').append(name).append(1, '=').append(*begin++);
        while (begin != end)
            data->append(1, ',').append(*begin++);
    }
}

} // namespace

#define FURITA_FILL_FIELDS(data, uid, mids) \
    (data)->append("uid=").append((std::to_string(uid))); \
    (data)->append("&mdb=pg"); \
    (data)->append("&ora_dsn=mail_dsn&source=furita"); \
    fill_field((data), "mids", (mids).begin(), (mids).end());

future<void> mops_call::move(
    const options& opt,
    const uint64_t& uid,
    const MidList& mids,
    const std::string& fid)
{
    ymod_http_client::string_ptr data(new std::string);
    FURITA_FILL_FIELDS(data, uid, mids);
    data->append("&dest_fid=").append(fid);
    return post_request(ctx_, httpClient_, opt, "complex_move", data, tvm_, headers_, tvmModule_);
}

future<void> mops_call::remove(
    const options& opt,
    const uint64_t& uid,
    const MidList& mids)
{
    ymod_http_client::string_ptr data(new std::string);
    FURITA_FILL_FIELDS(data, uid, mids);
    data->append("&nopurge=1");
    return post_request(ctx_, httpClient_, opt, "remove", data, tvm_, headers_, tvmModule_);
}

future<void> mops_call::mark(
    const options& opt,
    const uint64_t& uid,
    const MidList& mids,
    const std::string& status)
{
    ymod_http_client::string_ptr data(new std::string);
    FURITA_FILL_FIELDS(data, uid, mids);
    if (status == "RO")
        data->append("&status=read");
    else if (status == "New")
        data->append("&status=not_read");
    return post_request(ctx_, httpClient_, opt, "mark", data, tvm_, headers_, tvmModule_);
}

future<void> mops_call::label(
    const options& opt,
    const uint64_t& uid,
    const MidList& mids,
    const std::vector<std::string>& lids)
{
    ymod_http_client::string_ptr data(new std::string);
    FURITA_FILL_FIELDS(data, uid, mids);
    fill_field(data, "lids", lids.begin(), lids.end());
    return post_request(ctx_, httpClient_, opt, "label", data, tvm_, headers_, tvmModule_);
}

} // namespace furita
