#include "msearch_call.hpp"
#include <boost/bind.hpp>
#include <yamail/data/deserialization/yajl.h>
#include <yplatform/encoding/url_encode.h>
#include <ymod_httpclient/call.h>
#include <furita/common/http_headers.h>
#include <furita/common/logger.h>

namespace furita {

typedef yplatform::future::promise<void> promise_void;

class msearch_response_handler : public ymod_http_client::response_handler
{
public:
    msearch_response_handler(Envelopes& envs)
            : _data(std::string())
            , _envelopes(envs)
            , _ok(false)
            , _code(0)
    {}

    ymod_http_client::handler_version version() const override {
        return ymod_http_client::handler_version_already_parse_body;
    }

    void handle_data(const char* data, unsigned long long size) override {
        _data.append(data, size);
    }

    void handle_data_end() override {
        try {
            _envelopes = yamail::data::deserialization::fromJson<Envelopes>(_data);
            _ok = true;
        } catch (...) {}
    }

    Envelopes& envelopes() {
        return _envelopes;
    }

    void set_code(int code, const std::string &) override {
        _code = code;
    }

    int code() const override {
        return _code;
    }

    bool ok() const {
        return _ok;
    }

private:
    std::string _data;
    Envelopes& _envelopes;
    bool _ok;
    int _code;
};

namespace {

template <typename Future>
std::string exception_description(Future& f)
{
    if (!f.has_exception())
        return "no exceptions";
    try { f.get(); }
    catch (const yplatform::exception& e) { return e.public_message(); }
    catch (const std::exception& e) { return e.what(); }
    catch (...) {}
    return "unknown";
}

void handle_search(
    ymod_http_client::future_void_t f,
    boost::shared_ptr<msearch_response_handler> handler,
    MsearchCallPtr call,
    promise_void prom)
{
    if (f.has_exception()) {
        FURITA_LOG_ERROR(call->context(), logdog::message="msearch: status=error (" + exception_description(f) + ")")
        prom.set_exception(
            yplatform::exception("Error", "Failed to find messages"));
        return;
    }
    if (!handler->ok()) {
        FURITA_LOG_ERROR(call->context(), logdog::message="msearch: status=error (Bad response)")
        prom.set_exception(
            yplatform::exception("Error", "Failed to find messages"));
        return;
    }

    using namespace std::string_literals;
    FURITA_LOG_NOTICE(call->context(), logdog::message="msearch: status=ok"s + ", hits="+ std::to_string(call->mids().size()))

    prom.set();
}

} // namespace

future<void> MsearchCall::search(
    const std::string& suid,
    const std::string& uid,
    const std::string& query,
    std::size_t offset,
    std::size_t length,
    const std::string& folderSet,
    const std::string& remote_ip,
    bool /*log_pa*/)
{
    promise_void prom;

    std::stringstream req;
    req << "api/async/mail/furita";
    req << "?request=" << yplatform::url_encode(query);
    req << "&mdb=pg";
    if (!suid.empty())
        req << "&suid=" << suid;
    if (!uid.empty())
        req << "&uid=" << uid;
    req << "&imap=1";
    req << "&first=" << offset;
    if (length)
        req << "&count=" << length;
    if (!folderSet.empty()) {
        req << "&folder_set=" << folderSet;
    }

    if (!remote_ip.empty())
        req << "&remote_ip=" << yplatform::url_encode(remote_ip);
    req << "&noshared=1";
    req << "&exclude-trash=1";
    req << "&get=mid";

    using namespace std::string_literals;
    FURITA_LOG_DEBUG(ctx, logdog::message
        = "msearch: search"s + ", suid=" + suid +  ", uid=" + uid + ", text='" + query 
        + "'" + ", offset=" + std::to_string(offset) + ", length=" + std::to_string(length))

    std::string serviceTicket;
    const auto errorCode = tvmModule->get_service_ticket("msearch", serviceTicket);
    if (errorCode) {
        FURITA_LOG_ERROR(ctx, logdog::message="tvm error", logdog::error_code=errorCode);
        prom.set_exception(
            yplatform::exception("Error", "failed to obtain tvm ticket"));
        return prom;
    }

    auto handler = boost::make_shared<msearch_response_handler>(envelopes);

    auto futureMids = httpClient->get(ctx, url, timeouts, req.str(), make_headers(std::move(serviceTicket)), handler);
    futureMids.add_callback(boost::bind(
        handle_search, futureMids, handler, shared_from_this(), prom));

    return prom;
}

ymod_httpclient::headers_dict MsearchCall::make_headers(std::string serviceTicket) const {
    ymod_httpclient::headers_dict res = headers;
    if (!tvm.empty()) {
        res.emplace(HttpHeaderNames::ticket, tvm);
    }
    res.emplace(HttpHeaderNames::tvmServiceTicket, std::move(serviceTicket));
    return res;
}

} // namespace furita
