#pragma once

#include <furita/common/context.h>
#include <sharpei_client/http.h>
#include <boost/algorithm/string/join.hpp>

namespace furita {

using namespace sharpei::client::http;

class SharpeiHttpLoader: public HttpClient {
public:
    using ContextPtr = TContextPtr;

    explicit SharpeiHttpLoader(const ContextPtr& ctx, const ymod_httpclient::timeouts& timeouts)
        : ctx(ctx)
        , timeouts(timeouts)
    {}

    void aget(
        const Address& addr,
        Timeout /*timeout*/,
        const std::string& method,
        const Arguments& args,
        const Headers& /*headers*/,
        ResponseHandler handler,
        bool keepAlive,
        const std::string& /*requestId*/) const override
    {
        yhttp::options opts{false, false, keepAlive, timeouts};
        auto urlData = url(addr, method, args);
        auto req = yhttp::request::GET(urlData);
        auto httpClient = yplatform::find<yhttp::call>("http_client");
        httpClient->async_run(ctx, std::move(req), opts, [h = std::move(handler)](
                boost::system::error_code error, yhttp::response resp)
            {
                h(error, Response{static_cast<unsigned>(resp.status), std::move(resp.body)});
            }
        );
    }

    void apost(
        const Address& addr,
        Timeout /*timeout*/,
        const std::string& method,
        const Arguments& args,
        const Headers& /*headers*/,
        const std::string& data,
        ResponseHandler handler,
        bool keepAlive,
        const std::string& /*requestId*/) const override
    {
        yhttp::options opts{false, false, keepAlive, timeouts};
        auto urlData = url(addr, method, args);
        auto req = yhttp::request::POST(urlData, std::string(data));
        auto httpClient = yplatform::find<yhttp::call>("http_client");
        httpClient->async_run(ctx, std::move(req), opts, [h = std::move(handler)](
                boost::system::error_code error, yhttp::response resp)
            {
                h(error, Response{static_cast<unsigned>(resp.status), std::move(resp.body)});
            }
        );
    }

private:
    std::string url(const Address& addr, const std::string& method, const Arguments& args) const {
        std::vector<std::string> pairs;
        for (auto arg : args) {
            pairs.push_back(arg.first + "=" + boost::join(arg.second, ","));
        }
        std::stringstream urlStream;
        urlStream << addr.host << ":" << addr.port << method << "?" << boost::join(pairs, "&");
        return urlStream.str();
    }

    ContextPtr ctx;
    ymod_httpclient::timeouts timeouts;
};

}   // namespace furita
