#include "so_check_filter.h"
#include "make_search_helper.hpp"
#include <boost/fusion/adapted/struct/adapt_struct.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <mail/yreflection/include/yamail/data/serialization/yajl.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <ymod_httpclient/client.h>
#include <yplatform/log.h>
#include <furita/common/context.h>
#include <furita/common/logger.h>

namespace furita::processor::so_check {

void TRequest::add_condition(TFormField&& condition) {
    std::string condition_key = "condition" + std::to_string(++list_add_condition_number);
    form_fields.emplace(condition_key, condition);
}

void TRequest::add_action(TFormField&& action) {
    std::string action_key = "action" + std::to_string(++list_add_action_number);
    form_fields.emplace(action_key, action);
}

std::string get_url(const std::string& url_prefix, const std::string& uid) {
    std::string url = url_prefix;
    TCgiParameters cgi_params;
    cgi_params.emplace("id", uid);
    cgi_params.emplace("form_id", uid);
    if (url_prefix.find('?') == std::string::npos) {
        url += "?";
    } else {
        if (url.back() != '&') {
            url += "&";
        }
    }
    url += cgi_params();
    return url;
}

yplatform::future::future<EResult> so_check_filter(TContextPtr ctx, rules::rule_ptr rule,
                                                   std::shared_ptr<yhttp::cluster_client> cluster_client, const TParams &params) {
    static const std::string form_type = "create_filter";

    static const std::string statusMetric = "application_checkform_status_" + form_type;
    static const std::string resultMetric = "application_checkform_result_" + form_type;
    static const std::string timingsMetric = "application_checkform_timings_" + form_type;

    TRequest request_params_list;
    request_params_list.client_ip = params.user_ip;
    request_params_list.form_type = form_type;
    request_params_list.form_realpath = params.form_realpath;
    request_params_list.form_author = params.user_uid;
    request_params_list.subject = rule->name;

    for (auto&& condition : *rule->conditions) {
        const auto one_condition_rule = boost::make_shared<rules::rule>();
        one_condition_rule->conditions->emplace_back(condition);
        try {
            const auto value = furita::processor::msq::make_search_query(one_condition_rule, params.user_uid);
            TFormField condition_field{"string", "user", value};
            request_params_list.add_condition(std::move(condition_field));
        } catch(const std::runtime_error& ex) {}
    }

    for (auto&& action : *rule->actions) {
        const auto value = action->oper + ":" + action->param;
        TFormField action_field{"string", "user", value};
        request_params_list.add_action(std::move(action_field));
    }

    auto body = yamail::data::serialization::toJson(request_params_list).str();
    const auto url = get_url(params.url, ctx->uniq_id());
    auto http_request = yhttp::request::POST(url, "Content-Type: application/json", std::move(body));

    yplatform::future::promise<EResult> request_promise;

    cluster_client->async_run(ctx->CreateTaskContext(), http_request,  ymod_httpclient::simple_call::callback_type(
        [request_promise, ctx](boost::system::error_code ec, yhttp::response response) mutable {

            if(ec.value() == 0) {
                if (response.status == 200) {
                    try {
                        boost::property_tree::ptree response_parser;
                        std::stringstream stream(response.body);
                        boost::property_tree::read_xml(stream, response_parser);
                        if (response_parser.get<int>("spam")) {
                            request_promise.set(SPAM);
                        } else {
                            request_promise.set(OK);
                        }
                    } catch (const boost::property_tree::xml_parser::xml_parser_error &ex) {
                        FURITA_LOG_ERROR(ctx, logdog::message
                            = "so_check_filter: status=error, report:" + ex.message() + ", body:"
                            + boost::replace_all_copy(response.body.substr(0, 50), "\n", "\\n"))
                        request_promise.set(ERROR);
                    }
                } else {
                    FURITA_LOG_ERROR(ctx, logdog::message
                        = "so_check_filter: status=error, report:" + std::to_string(response.status) + ", body:"
                        + boost::replace_all_copy(response.body.substr(0, 50), "\n", "\\n"))
                    request_promise.set(ERROR);
                }
            }
            else{
                FURITA_LOG_ERROR(ctx, logdog::message="so_check_filter: status=error, report:" + ec.message())
                request_promise.set(ERROR);
            }

        }
    ));

    return request_promise;
}

}   // namespace furita::processor::so_check


BOOST_FUSION_ADAPT_STRUCT(
        furita::processor::so_check::TFormField,
        type,
        filled_by,
        value
)

BOOST_FUSION_ADAPT_STRUCT(
        furita::processor::so_check::TRequest,
        client_ip,
        form_type,
        form_realpath,
        form_author,
        subject,
        form_fields
)
