#pragma once

#include <boost/system/error_code.hpp>

namespace furita::tupita {

enum class EError {
    Ok,
    InvalidResponse,
    RequestParseError,
    TvmServiceTicketError,
    ExceptionWasThrown,
    Unknown
};

}

namespace boost::system {

template <>
struct is_error_code_enum<furita::tupita::EError> : std::true_type {};

}

namespace furita::tupita {

class TErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "furita_tupita_converter";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::InvalidResponse:
                return "Invalid Response";
            case EError::RequestParseError:
                return "Incorrect request";
            case EError::TvmServiceTicketError:
                return "Error obtaining TVM ticket";
            case EError::ExceptionWasThrown:
                return "Exception was thrown";
            case EError::Unknown:
                break;
        }
        return "Unknown error code: " + std::to_string(value);
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

}
