#include "http_client.h"

#include <mail/furita/include/furita/common/logger.h>
#include <mail/furita/include/furita/common/tvm.hpp>
#include <mail/yreflection/include/yamail/data/deserialization/yajl.h>

namespace furita::tupita {

void TTupitaClientImpl::HandleResponse(TContextPtr ctx, TErrorCode errorCode, const yhttp::response& response,
     TTupitaClientCallback callback
) {
    if (errorCode) {
        FURITA_LOG_ERROR(ctx, logdog::error_code=errorCode, logdog::message="error occurred");
        return callback(errorCode, {});
    }
    const auto& responseStr = response.body;
    const auto httpStatus = response.status;
    auto conversionError = tupita::EError::Unknown;
    std::string errorMessage;
    TTupitaResponseOk responseOk;
    TTupitaResponseError responseError;
    try {
        switch (httpStatus) {
            case 200:
                responseOk = yamail::data::deserialization::fromJson<TTupitaResponseOk>(responseStr);
                std::tie(conversionError, errorMessage) = OnParsedOk(responseOk);
                break;
            default:
                FURITA_LOG_NOTICE(ctx, logdog::message="tupita converter responded with error");
                responseError = yamail::data::deserialization::fromJson<TTupitaResponseError>(responseStr);
                std::tie(conversionError, errorMessage) = OnParsedError(responseError);
        }
    } catch (const std::exception& e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="error parsing tupita converter answer");
        std::tie(conversionError, errorMessage) = OnInvalidResponse("Exception was thrown");
    }
    callback(conversionError, {GetQueries(responseOk), errorMessage});
}

std::vector<std::string> TTupitaClientImpl::GetQueries(const TTupitaResponseOk& responseOk) {
    auto transformer = [](const reflection::TQuery& query){ return query.query; };
    auto range = responseOk.conditions | boost::adaptors::transformed(std::move(transformer));
    return {range.begin(), range.end()};
}

TTupitaClientImpl::TErrorInfo TTupitaClientImpl::OnParsedOk(const TTupitaResponseOk& response) {
    if (response.status == "ok" && !response.conditions.empty()) {
        return {tupita::EError::Ok, {}};
    } else {
        return OnInvalidResponse();
    }
}

TTupitaClientImpl::TErrorInfo TTupitaClientImpl::OnParsedError(const TTupitaResponseError& response) {
    if (response.status == "error" && !response.error.conditions.empty()) {
        std::stringstream errorMessage;
        errorMessage << response.error.error_type << ":";
        for (const auto& cond : response.error.conditions) {
            errorMessage << " [" << cond.orig_index << "] " << cond.message;
        }
        return {tupita::EError::RequestParseError, errorMessage.str()};
    } else {
        return OnInvalidResponse();
    }
}

TTupitaClientImpl::TErrorInfo TTupitaClientImpl::OnInvalidResponse(const std::string& errorMsg) {
    return {tupita::EError::InvalidResponse, errorMsg};
}

void TTupitaClientImpl::DoRequest(TContextPtr ctx, const TTupitaRequest& request, TTupitaClientCallback cb) {
    if (request.Conditions.empty()) {
        return cb(tupita::EError::Ok, {});
    }
    std::string reqBody = ComposeRequestBody(request.Conditions);
    std::string ticket = GetTvmServiceTicket(ctx);
    if (ticket.empty()) {
        return cb(tupita::EError::TvmServiceTicketError, {});
    }

    auto callback = [ctx, cb = std::move(cb), executor = IoContext.get_executor()](auto ec, auto response){
        boost::asio::post(executor, 
            std::bind(&TTupitaClientImpl::HandleResponse, ctx, std::move(ec), std::move(response), std::move(cb))
        );
    };

    auto httpRequest = ymod_httpclient::request::POST(
        ComposeRequestUrl(request.Domain, request.OrgId),
        MakeTvmHeader(std::move(ticket)),
        std::move(reqBody)
    );
    ClusterCall->async_run(ctx->CreateTaskContext(), std::move(httpRequest), std::move(callback));
}

std::string TTupitaClientImpl::ComposeRequestUrl(const std::optional<std::string>& domain,
    const std::optional<std::string>& orgId
) {
    std::string url{"/api/mail/conditions/convert"};
    auto params = yhttp::url_encode({
        {"domain", domain.value_or(std::string{})},
        {"org_id", orgId.value_or(std::string{})},
    });
    return url + params;
}

std::string TTupitaClientImpl::ComposeRequestBody(const TJsonVector& conditions) {
    json_value body;
    body["conditions"].set_array();
    auto condArray = body["conditions"];
    for (const auto& c : conditions) {
        condArray.push_back(c);
    }
    return body.stringify();
}

std::string TTupitaClientImpl::GetTvmServiceTicket(TContextPtr ctx) const {
    std::string ticket;
    const auto errorCode = TvmModule->get_service_ticket(TUPITA_SERVICE_NAME, ticket);
    if (errorCode) {
        FURITA_LOG_ERROR(ctx, logdog::error_code = errorCode, logdog::message = "Error obtaining TVM ticket");
        return {};
    }

    return ticket;
}

}
