#pragma once

#include "error_code.h"

#include <mail/furita/include/furita/common/context.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_tvm/module.h>

#include <boost/asio/io_context.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <yplatform/json.h>


BOOST_FUSION_DEFINE_STRUCT((furita)(tupita)(reflection), TQuery,
    (std::string, query)
);

BOOST_FUSION_DEFINE_STRUCT((furita)(tupita)(reflection), TTupitaResponseOk,
    (std::string, status)
    (std::vector<furita::tupita::reflection::TQuery>, conditions)
);

BOOST_FUSION_DEFINE_STRUCT((furita)(tupita)(reflection), TCondition,
    (std::string, message)
    (int, orig_index)
);

BOOST_FUSION_DEFINE_STRUCT((furita)(tupita)(reflection), TError,
    (std::string, error_type)
    (std::vector<furita::tupita::reflection::TCondition>, conditions)
);

BOOST_FUSION_DEFINE_STRUCT((furita)(tupita)(reflection), TTupitaResponseError,
    (std::string, status)
    (furita::tupita::reflection::TError, error)
);

namespace furita::tupita {

using reflection::TTupitaResponseOk;
using reflection::TTupitaResponseError;

using yplatform::json_value;
using TJsonVector = std::vector<json_value>;
using TErrorCode = ::boost::system::error_code;

const std::string TUPITA_SERVICE_NAME{"msearch"};

struct TTupitaRequest {
    TJsonVector Conditions;
    std::optional<std::string> Domain;
    std::optional<std::string> OrgId;
};

struct TTupitaResponse {
    std::vector<std::string> Queries;
    std::string ErrorMessage;
};

using TTupitaClientCallback = std::function<void(TErrorCode, TTupitaResponse)>;

struct ITupitaClient {
    virtual ~ITupitaClient() = default;
    virtual void DoRequest(TContextPtr ctx, const TTupitaRequest& request,
        TTupitaClientCallback callback) = 0;
};

class TTupitaClientImpl : public ITupitaClient {
    using TClusterCall = ymod_httpclient::cluster_call;
    using TClusterCallPtr = std::shared_ptr<TClusterCall>;
    using TTvmModulePtr = std::shared_ptr<ymod_tvm::tvm2_module>;
    using TErrorInfo = std::tuple<tupita::EError, std::string>;

public:
    TTupitaClientImpl(TClusterCallPtr clusterCall, TTvmModulePtr tvmModule, boost::asio::io_context& ioContext)
        : ClusterCall(clusterCall)
        , TvmModule(tvmModule)
        , IoContext(ioContext)
    {
    }

    void DoRequest(TContextPtr ctx, const TTupitaRequest& request,
        TTupitaClientCallback clientCallback) override;

private:
    static std::vector<std::string> GetQueries(const TTupitaResponseOk& responseOk);
    static TErrorInfo OnParsedOk(const TTupitaResponseOk& response);
    static TErrorInfo OnParsedError(const TTupitaResponseError& response);
    static TErrorInfo OnInvalidResponse(const std::string& errorMsg = {});

    static std::string ComposeRequestUrl(const std::optional<std::string>& domain,
        const std::optional<std::string>& orgId);
    static std::string ComposeRequestBody(const TJsonVector& conditions);
    static void HandleResponse(TContextPtr ctx, TErrorCode code, const yhttp::response& response, TTupitaClientCallback callback);
    std::string GetTvmServiceTicket(TContextPtr ctx) const;

private:
    TClusterCallPtr ClusterCall;
    TTvmModulePtr TvmModule;
    boost::asio::io_context& IoContext;
};

using TTupitaClientPtr = std::shared_ptr<ITupitaClient>;

}
