#! /usr/bin/env python

import requests
import logging
import argparse
import base64
from xml.dom.minidom import xml
from email import Message


def download_translations(oauth_token):
    try:
        url = 'https://tanker-api.yandex-team.ru' +\
            '/projects/export/xml/?project-id=yfurita' +\
            '&language=ru,en,uk,tr&status=unapproved&safe'
        headers = {'Authorization': 'OAuth ' + oauth_token}
        r = requests.get(url, timeout=1.0, headers=headers,
                         verify=False)  # XXX: verification not works in older python
    except:
        logging.exception('will not update default templates')
        return False, ""
    return r.status_code == requests.codes.ok, r.content


class DataParser:

    def __init__(self):
        self.texts = dict()
        self.subjects = dict()
        self.froms = dict()
        pass

    def parse(self, data):
        self.texts.clear()
        self.subjects.clear()
        self.froms.clear()

        try:
            domtree = xml.dom.minidom.parseString(data)
        except:
            return
        self.head = domtree.documentElement

        keysets = self.head.getElementsByTagName('keyset')
        for keyset in keysets:
            keys = keyset.getElementsByTagName('key')

            for k in keys:
                if not k.hasAttribute('id'):
                    continue
                self.key_processing(k)

    def key_processing(self, key):
        if not key.hasAttribute('id'):
            return
        key_chunks = key.getAttribute('id').split('_')

        if len(key_chunks) == 0:
            return

        values = key.getElementsByTagName('value')
        for val in values:
            if not val.hasAttribute('language'):
                continue

            name = val.getAttribute('language')
            for el in reversed(key_chunks[1:]):
                name = el + '_' + name

            if len(val.childNodes) > 0:
                if key_chunks[0] == 'from':
                    self.froms[name] = (val.childNodes[0].data).encode('utf-8')
                elif key_chunks[0] == 'subject':
                    self.subjects[name] = (val.childNodes[0].data).encode('utf-8')
                elif key_chunks[0] == 'confirm':
                    self.texts[name] = (val.childNodes[0].data).encode('utf-8')


class MessageCreator():

    def __init__(self):
        self.msg = Message.Message()
        self.msg.add_header('To', '$raddr')
        self.msg.add_header('Content-Transfer-Encoding', '8bit')
        self.msg.set_charset('UTF-8')
        self.msg.set_type('text/html')

    # from, subject, body - utf-8
    def create(self, From, Subject, Body, filename):
        self.msg.set_payload(Body)

        if self.msg.get('Subject'):
            self.msg.replace_header('Subject', Subject)
        else:
            self.msg.add_header('Subject', Subject)

        if self.msg.get('From'):
            self.msg.replace_header('From', From)
        else:
            self.msg.add_header('From', From)

        with open(filename, 'w') as f:
            f.write(str(self.msg))


class MessageHelper:
    def __init__(self, texts, subjects, froms):
        self.texts = texts.copy()
        self.subjects = subjects.copy()
        self.froms = froms.copy()
        self.prefix = '=?UTF-8?B?'
        self.suffix = '?='
        self.default_value = ''

    def get_subject(self, action_lang, default='forward_ru'):
        result = self.default_value
        if action_lang in self.subjects:
            result = self.subjects.get(action_lang)
        elif default in self.subjects:
            result = self.subjects.get(default)
        result = '{0}{1}{2}'.format(self.prefix,
                                    base64.b64encode(result),
                                    self.suffix)
        return result

    def get_from(self, lang, default='ru'):
        result = self.default_value
        if lang in self.froms:
            result = self.froms.get(lang)
        elif default in self.froms:
            result = self.froms.get(default)
        result = '{0}{1}{2} <$envelopeFrom>'.format(self.prefix,
                                                    base64.b64encode(result),
                                                    self.suffix)
        return result

    def get_body(self, action_lang, default='forward_ru'):
        if action_lang in self.texts:
            return self.texts.get(action_lang)
        if default in self.texts:
            return self.texts.get(default)
        return self.default_value

    def identify_lang(self, action_lang, default='ru'):
        pos = action_lang.rfind('_')
        if pos != -1:
            return action_lang[pos + 1:]
        return default


def main():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter,
                                     description="Download furita templates from Tanker")
    parser.add_argument("-t", "--token", required=True,
                        help="OAuth token for requesting Tanker API")
    parser.add_argument("-p", "--pattern", default="../etc/filter_confirm_{}.eml",
                        help="Templates path pattern (substring '{}' will be replaced with language code)")
    args = parser.parse_args()
    run(args)


def run(args):
    success, data = download_translations(args.token)
    if success:
        parser = DataParser()
        parser.parse(data)

        helper = MessageHelper(parser.texts, parser.subjects, parser.froms)

        msg_creator = MessageCreator()

        for a_l in helper.texts.keys():
            lang = helper.identify_lang(a_l)

            From = helper.get_from(lang)
            Subject = helper.get_subject(a_l)
            Body = helper.get_body(a_l)

            filename = args.pattern.format(a_l)
            msg_creator.create(From, Subject, Body, filename)


if __name__ == "__main__":
    main()
