from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPath,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
)
import os
import yaml
import json
import logging
import mail.unistat.cpp.cython.logs as logs
import mail.furita.unistat.cpp.run as run


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


SERVICES = [
    ['sharpei',         '/conninfo'],
    ['mops',            '/complex_move', '/remove', '/mark', '/label'],
    ['msearch',         '/api/async/mail/furita'],
    ['tupita',          '/api/mail/conditions/convert'],
]

SERVICE_ENDPOINTS = [
    '/api/blackwhitelist',
    '/api/blacklist.json',
    '/api/blacklist_add.json',
    '/api/blacklist_remove.json',
    '/api/whitelist.json',
    '/api/whitelist_add.json',
    '/api/whitelist_remove.json',

    '/api/edit.json',
    '/api/verify.json',
    '/api/enable.json',
    '/api/order.json',
    '/api/remove.json',

    '/api/apply.json',
    '/api/preview.json',

    '/api/list.json',
    '/api/multi_list',

    '/v1/domain/rules/get',
    '/v1/domain/rules/set',
]


def make_access_log_meters(name_prefix, endpoints, http_hist_buckets):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist(http_hist_buckets, 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPath(endpoint, name_prefix),
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath(http_hist_buckets, endpoint, 'access_log_request'),
        ]

    return meters


def make_http_client_log_meters(services, http_hist_buckets):
    meters = [
        HttpClientHttpRequestTotalTimeHist(http_hist_buckets, handler, service[0] + handler.replace('/', '_') + "_time")
        for service in services
        for handler in service[1:]
    ]
    meters += [
        HttpClientHttpRequestCountByStatus(handler, "count_by_status_" + service[0])
        for service in services
        for handler in service[1:]
    ]

    return meters


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='', help='path for unistat.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def make_service_config(data):
    return ServiceConfig(
        access_log=os.path.join(os.curdir, data['config']['log']['access_log']['sinks']['path']),
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks']['path']),
    )


ServiceConfig = namedtuple('ServiceConfig', (
    'access_log',
    'httpclient_log',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.service_config) as f:
        service_config = make_service_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    http_hist_buckets = (0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 16, 18, 20, 25, 30, 35, 40, 50, 100, 200, 500, 1000, 2000, 3000)

    access_log = logs.AccessTskv([], make_access_log_meters("access_log", SERVICE_ENDPOINTS, http_hist_buckets), fast_forward, service_config.access_log)
    http_client_log = logs.HttpClientLog([], make_http_client_log_meters(SERVICES, http_hist_buckets), fast_forward, service_config.httpclient_log)

    logs_list = [access_log, http_client_log]

    run.run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
