#include <mail/furita/ymod_db/include/repository.h>

#include <mail/furita/include/furita/core/log.h>
#include <mail/furita/include/furita/core/types.h>
#include <mail/furita/ymod_db/include/internal/types.h>
#include <mail/webmail/commondb/include/settings.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <mail/webmail/commondb/include/with_executers.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/deserialization/ptree.h>
#include <yplatform/module_registration.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>

#include <string>


BOOST_FUSION_DEFINE_STRUCT((furita), Settings,
    (std::string, reactor)
    (std::string, logger)
)


namespace furita {

namespace {

auto fromReflection(reflection::Revision& data) {
    return furita::Revision(data.revision);
}

auto fromReflection(reflection::DomainRules& data) {
    return furita::DomainRules {
        .orgId = OrgId(data.org_id),
        .revision = Revision(data.revision),
        .rules = Rules(data.rules)
    };
}

auto fromReflection(reflection::OptionalDomainRules& data) {
    io_result::Optional<furita::DomainRules> s;

    if (data.domainRules) {
        s = fromReflection(*data.domainRules);
    }
    return s;
}

auto makeRequestInfo(const SetDomainRulesParams& params) {
    reflection::RequestInfo requestInfo;
    requestInfo.org_id = params.orgId;
    requestInfo.request_id = params.uniqId;
    return requestInfo;
}

} // namespace


struct Database : public commondb::WithExecuters, public Repository, public yplatform::module {

    virtual ~Database() = default;

    void init(const yplatform::ptree& cfg) {
        common = yamail::data::deserialization::fromPtree<Settings>(cfg);
        pg = commondb::readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());
    }

    using WithExecuters::anyExecutor;
    using WithExecuters::readWriteExecutor;

    template<class Params>
    auto master(const Params& params) const {
        auto ctxLogger = getContextLogger(params);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(params.orgId.t), params.uniqId, "furitadb");
        return readWriteExecutor(params.uniqId, logger, profiler);
    }

    template<class Params>
    auto anything(const Params& params) const {
        auto ctxLogger = getContextLogger(params);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(params.orgId.t), params.uniqId, "furitadb");
        return anyExecutor(params.uniqId, logger, profiler);
    }

    void asyncSetDomainRules(const SetDomainRulesParams& params, OnSetDomainRules handler) const override {
        master(params)->request(
            pgg::withQuery<query::SetDomainRules>(
                query::RequestInfo(makeRequestInfo(params)),
                query::Rules(params.rules)
            ),
            std::move(handler),
            [] (reflection::Revision data) { return fromReflection(data); }
        );
    }

    void asyncGetDomainRules(const GetDomainRulesParams& params, OnGetDomainRules handler) const override {
        anything(params)->request(
            pgg::withQuery<query::GetDomainRules>(
                query::OrgId(params.orgId)
            ),
            std::move(handler),
            [] (reflection::OptionalDomainRules data) { return fromReflection(data); }
        );
    }

    Settings common;
    commondb::PgSettings pg;
};

} // namespace furita


DEFINE_SERVICE_OBJECT(furita::Database)
