import logging

logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")

from datetime import timedelta
from mail.ghost.lib.calculator import IResourceCalculator
from mail.ghost.lib.utils import total_days
from mail.ghost.bin.statface_proxy import StatfaceProxy
from mail.ghost.bin.utils import get_launch_args, get_available_calculators, get_calculator_name


def show_available_calculators():
    print(*get_available_calculators(), sep="\n")


def get_metrics(calculator, time_interval):
    data = calculator.get_resources()
    days = total_days(time_interval)

    result = []
    for index in range(0, days):
        day = time_interval[0] + timedelta(days=index)

        any_element = next(iter(data.items()))[1]
        groups = set().union(*(v.keys() for v in data.values())) if isinstance(any_element, dict) else [None]
        for group in groups:
            values = {'fielddate': str(day)}
            if group:
                values['group'] = group
                for key in data:
                    values[key] = data[key][group][index] if group in data[key] else 0.0
            else:
                for key in data:
                    values[key] = data[key][index]
            result += [values]

    assert len(result) == days * len(groups)
    return result


def run_calculators(calculators, dry_run, time_interval):
    statface_proxy = StatfaceProxy()
    logger = logging.getLogger(__name__)
    logger.info("stats will be collected for the interval [{begin}, {end})".format(begin=time_interval[0], end=time_interval[1]))

    for calculator_type in calculators:
        try:
            calculator_name = get_calculator_name(calculator_type)
            calculator = calculator_type()

            logger.info("calculator={} started running...".format(calculator_name))

            data = get_metrics(calculator, time_interval)

            logger.info("calculator={} data={}".format(calculator_name, data))

            if not dry_run:
                project_path = calculator.get_statface_project_path()

                if project_path:
                    response = statface_proxy.send_json_data(data, project_path)
                    logger.info("Statface response: {}".format(response.text))
                else:
                    logger.info("calculator={} skipped because Statface path isn't set".format(calculator_name))

        except Exception:
            logger.exception('An exception occurred while running {}'.format(calculator_name))


def main():
    launch_args = get_launch_args()

    IResourceCalculator.dry_run = launch_args.dry_run
    IResourceCalculator.time_interval = launch_args.time_interval

    if launch_args.mode == "run":
        run_calculators(launch_args.calculators, launch_args.dry_run, launch_args.time_interval)
    else:
        show_available_calculators()


if __name__ == "__main__":
    main()
