import argparse
from datetime import datetime, timedelta
from mail.ghost.calculators import get_calculators


class LaunchArgs(object):
    def __init__(self, mode, calculators, dry_run, time_interval):
        self.mode = mode
        self.calculators = calculators
        self.dry_run = dry_run
        self.time_interval = time_interval


def get_calculator_name(calculator):
    return ".".join([calculator.__module__, calculator.__name__])


def get_available_calculators():
    return [get_calculator_name(calculator) for calculator in get_calculators()]


def get_calculators_by_names(calculators_names):
    calculators_dict = dict((get_calculator_name(calculator), calculator) for calculator in get_calculators())
    return [calculators_dict[calculator_name] for calculator_name in calculators_names]


class TimeIntervalConstructor(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        assert len(values) == 2, "expected 2 values: begin_delta, end_delta"
        now = datetime.now().date()
        time_interval = (now - timedelta(days=values[0]), now - timedelta(days=values[1]))
        assert time_interval[0] < time_interval[1]
        setattr(namespace, self.dest, time_interval)


def get_launch_args():
    yesterday_today = (datetime.now().date() - timedelta(days=1), datetime.now().date())

    parser = argparse.ArgumentParser(description="Resource utilization calculators runner")
    parser.add_argument("-l", "--list", dest="show_calculators", action="store_true", help="show available calculators")
    parser.add_argument("-r", "--run", dest="calculators_names", nargs='+', metavar="calculator-name",
                        choices=get_available_calculators(), help="set calculators to run")
    parser.add_argument("-dr", "--dry-run", dest="dry_run", action="store_true",
                        help="run calculators without sending data to statface")
    parser.add_argument("-i", "--time_interval", dest="time_interval", nargs='+', type=int, default=yesterday_today,
                        action=TimeIntervalConstructor,
                        help="two integers begin_delta, end_delta specifying the interval over which statistics will be collected. "
                             "Default value is \"1 0\" (without quotes) which corresponds to the interval [yesterday, today)."
                             "Be aware that in some cases only the current value of resource quota is known and that value will be reported for all requested days.")
    args = parser.parse_args()
    launch_args = LaunchArgs("show" if args.show_calculators else "run",
                             get_calculators_by_names(
                                 args.calculators_names) if args.calculators_names else get_calculators(),
                             args.dry_run,
                             args.time_interval)
    return launch_args
