import re

from mail.ghost.lib.register import register
from mail.ghost.lib.calculator import IResourceCalculator
from mail.ghost.lib.utils import sum_elementwise, total_days
from mail.ghost.calculators.mail.utils.mail_qloud_proxy import MailQloudProxy
from mail.ghost.calculators.mail.utils.utils import make_golovan_context


@register
class MailQloudCpuCalculator(IResourceCalculator):
    def __init__(self):
        self.qloud_proxy = MailQloudProxy()

    def get_used_cpu(self):
        environments = self.qloud_proxy.get_environments()
        total_used_cpu = [0] * total_days(self.time_interval)

        for environment in environments:
            value = self.get_resource_utilization_from_golovan("QLOUD", make_golovan_context("qloud_used_cpu", environment))
            total_used_cpu = sum_elementwise(total_used_cpu, value)

        return total_used_cpu

    def get_cpu_limit(self):
        return [self.qloud_proxy.get_cpu_quota()] * total_days(self.time_interval)

    def get_resources(self):
        resources = {
            "used": self.get_used_cpu(),
            "quota": self.get_cpu_limit()
        }
        return resources

    def get_statface_project_path(self):
        return "Mail/QLOUD/CPU"


@register
class MailQloudCpuExCalculator(IResourceCalculator):
    def __init__(self):
        self.qloud_proxy = MailQloudProxy()

    def _get_service_group(self, environment):
        m = re.search(r"^mail\.(?P<service>[^.]+)\..*$", environment)
        return m["service"] if m else "<other>"

    def get_used_cpu(self):
        environments = self.qloud_proxy.get_environments()

        service_used_cpu = {}
        for environment in environments:
            service_group = self._get_service_group(environment)
            used_cpu = service_used_cpu.get(service_group, [0] * total_days(self.time_interval))
            value = self.get_resource_utilization_from_golovan("QLOUD", make_golovan_context("qloud_used_cpu", environment))
            service_used_cpu[service_group] = sum_elementwise(used_cpu, value)

        return service_used_cpu

    def get_cpu_limit(self):
        environment_quota = self.qloud_proxy.get_cpu_quota_by_env()

        service_quota_cpu = {}
        for environment, quotas in environment_quota.items():
            service_group = self._get_service_group(environment)
            quota_cpu = service_quota_cpu.get(service_group, [0] * total_days(self.time_interval))
            value = [sum(float(v["quota"]) for v in quotas)] * total_days(self.time_interval)
            service_quota_cpu[service_group] = sum_elementwise(quota_cpu, value)

        return service_quota_cpu

    def get_resources(self):
        resources = {
            "used": self.get_used_cpu(),
            "quota": self.get_cpu_limit()
        }
        return resources

    def get_statface_project_path(self):
        return "Mail/QLOUD/CPU_EX"
