import re

from mail.ghost.lib.register import register
from mail.ghost.lib.calculator import IResourceCalculator
from mail.ghost.lib.utils import sum_elementwise, total_days
from mail.ghost.calculators.mail.utils.mail_qloud_proxy import MailQloudProxy
from mail.ghost.calculators.mail.utils.utils import make_golovan_context


@register
class MailQloudMemoryCalculator(IResourceCalculator):
    def __init__(self):
        self.qloud_proxy = MailQloudProxy()

    def get_used_memory(self):
        environments = self.qloud_proxy.get_environments()
        total_used_memory = [0] * total_days(self.time_interval)

        for environment in environments:
            value = self.get_resource_utilization_from_golovan("QLOUD", make_golovan_context("qloud_used_mem", environment))
            total_used_memory = sum_elementwise(total_used_memory, value)

        return total_used_memory

    def get_memory_quota(self):
        return [self.qloud_proxy.get_memory_quota()] * total_days(self.time_interval)

    def get_resources(self):
        resources = {
            "used": self.get_used_memory(),
            "quota": self.get_memory_quota()
        }
        return resources

    def get_statface_project_path(self):
        return "Mail/QLOUD/RAM"


@register
class MailQloudMemoryExCalculator(IResourceCalculator):
    def __init__(self):
        self.qloud_proxy = MailQloudProxy()

    def _get_service_group(self, environment):
        m = re.search(r"^mail\.(?P<service>[^.]+)\..*$", environment)
        return m["service"] if m else "<other>"

    def get_used_memory(self):
        environments = self.qloud_proxy.get_environments()

        service_used_memory = {}
        for environment in environments:
            service_group = self._get_service_group(environment)
            used_memory = service_used_memory.get(service_group, [0] * total_days(self.time_interval))
            value = self.get_resource_utilization_from_golovan("QLOUD", make_golovan_context("qloud_used_mem", environment))
            service_used_memory[service_group] = sum_elementwise(used_memory, value)

        return service_used_memory

    def get_memory_limit(self):
        environment_quota = self.qloud_proxy.get_memory_quota_by_env()

        service_quota_memory = {}
        for environment, quotas in environment_quota.items():
            service_group = self._get_service_group(environment)
            quota_memory = service_quota_memory.get(service_group, [0] * total_days(self.time_interval))
            value = [sum(float(v["quota"]) for v in quotas)] * total_days(self.time_interval)
            service_quota_memory[service_group] = sum_elementwise(quota_memory, value)

        return service_quota_memory

    def get_resources(self):
        resources = {
            "used": self.get_used_memory(),
            "quota": self.get_memory_limit()
        }
        return resources

    def get_statface_project_path(self):
        return "Mail/QLOUD/RAM_EX"
