from datetime import datetime, time, timedelta
from mail.ghost.lib.golovan_proxy import GolovanProxy
from mail.ghost.lib.solomon_proxy import SolomonProxy
from mail.ghost.lib.utils import total_days


def date_to_datetime(date):
    return datetime.combine(date, time())


def total_hours(timedelta):
    hours = int(timedelta.total_seconds() // 3600)
    assert hours >= 0
    return hours


def get_nth_percentile(data, n):
    sorted_data = sorted(data)
    assert 0 <= n < 100
    index = (len(sorted_data) * n) // 100
    assert index < len(sorted_data)
    return sorted_data[index]


def aggregate_data(data, begin, end):
    days = (end - begin).days
    aggregated = [0] * days
    values = [value for timestamp, value in data]
    if None in values:
        # TODO(MAILDLV-3691): raise an exception here. It is likely indicates that we requested info about a host not known to Golovan (see explicit try&catch in DiskMachine*Calculator)
        return aggregated
    POINTS_IN_DAY = 24 * 12  # values are reported for each 5 min interval
    assert len(values) == days * POINTS_IN_DAY
    for day in range(0, days):
        this_day_beginning = day * POINTS_IN_DAY
        this_day_data = values[this_day_beginning: this_day_beginning + POINTS_IN_DAY]
        DEFAULT_PERCENTILE = 99  # most tough 15 minutes are discarded
        aggregated[day] = get_nth_percentile(this_day_data, DEFAULT_PERCENTILE)

    return aggregated


def skip_midnight_of_the_last_day(last_day):
    return date_to_datetime(last_day) - timedelta(microseconds=1)


def get_resource_utilization_from_golovan(host, context, time_interval):
    """
    Возвращает утилизацию ресурса из голована.
    Берется утилизация ресурса в интервале времени за предыдущий день с шагом в 5 минут.
    В каждом часу берется максимальное значение.
    По часам берется среднее.
    :param host: хост в головане
    :param context: контекст в головане
    """

    left_time_bound_timestamp = date_to_datetime(time_interval[0]).timestamp()
    right_time_bound_timestamp = skip_midnight_of_the_last_day(time_interval[1]).timestamp()

    POINTS_IN_DAY = 24 * 12  # values are reported for each 5 min interval
    load_segments = POINTS_IN_DAY * total_days(time_interval)

    values = GolovanProxy().get_values(host, timedelta(minutes=5).total_seconds(),
                                       left_time_bound_timestamp, right_time_bound_timestamp, context, load_segments)
    return aggregate_data(values, time_interval[0], time_interval[1])


def get_resource_utilization_from_solomon(cgi_params, time_interval):
    """
    Возвращает утилизацию ресурса из соломона.
    Берется утилизация ресурса в интервале времени за предыдущий день с шагом в 5 минут.
    В каждом часу берется максимальное значение.
    По часам берется среднее.
    :param cgi_params: мапа cgi параметров запроса в api соломона, указание временного интервала игнорируется
    """

    # begin timestamp
    cgi_params["b"] = date_to_datetime(time_interval[0]).isoformat() + "Z"
    # end timestamp
    cgi_params["e"] = skip_midnight_of_the_last_day(time_interval[1]).isoformat() + "Z"
    # time step 5m
    cgi_params["grid"] = "5m"

    values = SolomonProxy().get_values(cgi_params)
    return aggregate_data(values, time_interval[0], time_interval[1])
