import requests
import json
from mail.ghost.lib.config import QloudProxyConfig


class QloudProxy(object):
    def __init__(self):
        self.config = QloudProxyConfig()

    def get_environments(self, project_name):
        response = requests.get(
            "{}/api/v1/project/{}".format(self.config.host, project_name),
            headers={"Authorization": "OAuth {}".format(self.config.oauth_token)}
        )
        response = json.loads(response.text)["applications"]
        environments = []
        for application in response:
            for environment in application["environments"]:
                environments.append(environment["objectId"])
        return environments

    def _request_quota(self, project_name):
        return requests.get(
            "{}/api/v1/project/{}/quota".format(self.config.host, project_name),
            headers={"Authorization": "OAuth {}".format(self.config.oauth_token)}
        )

    def get_resource_quota(self, project_name, resource_name):
        response = self._request_quota(project_name)
        quota_parts = json.loads(response.text)["quota"]

        quota = 0
        for part in quota_parts:
            if part["resource"] == resource_name and part["hardwareSegment"] in ["mail", "personal"]:
                quota += part["quota"]

        return quota

    def get_resource_used_quota_by_env(self, project_name, resource_name):
        response = self._request_quota(project_name)
        quota_parts = json.loads(response.text)["usedQuota"]

        def quota_filter(x):
            return (
                x["resource"] == resource_name and
                x["hardwareSegment"] in ["mail", "personal"] and
                x["location"] == "WHOLE"
            )

        return {
            env: list(filter(quota_filter, quotas))
            for env, quotas in quota_parts.items()
        }

    def get_cpu_quota(self, project_name):
        return self.get_resource_quota(project_name, "CPU")

    def get_memory_quota(self, project_name):
        return self.get_resource_quota(project_name, "MEMORY_GB")

    def get_cpu_quota_by_env(self, project_name):
        return self.get_resource_used_quota_by_env(project_name, "CPU")

    def get_memory_quota_by_env(self, project_name):
        return self.get_resource_used_quota_by_env(project_name, "MEMORY_GB")
