from collections import namedtuple

from concurrent.futures import ThreadPoolExecutor

from .utils import retry_session
import base64
import logging

log = logging.getLogger(__name__)
Attach = namedtuple('Attach', ('content_type', 'content'))
Envelope = namedtuple('Envelope', ('from_email', 'cc', 'message_id'))
CalendarEvent = namedtuple('CalendarEvent', ('users', "envelope", 'attach'))


class CalendarError(Exception):
    pass


class Calendar(object):
    def __init__(self, url, session=retry_session(), timeout=20):
        self.url = url
        self.session = session
        self.timeout = timeout
        self.pool = ThreadPoolExecutor(max_workers=25)

    def mailhook(self, calendar_event, mid=None, hid=None, stid=None):
        def post(user):
            json_payload = {
                'content-type': calendar_event.attach.content_type,
                'message-id': calendar_event.envelope.message_id,
                'yandex-uid': user.uid,
                'to': user.email,
                'cc': ','.join(calendar_event.envelope.cc),
                'from': calendar_event.envelope.from_email,
                'mid': mid,
                'hid': hid,
                'stid': stid,
                'calendar-data': base64.standard_b64encode(calendar_event.attach.content)
            }
            return self.session.post(
                self.url,
                json=json_payload,
                timeout=self.timeout,
            )

        submissions = self.pool.map(post, calendar_event.users)

        filtered = filter(lambda submission: submission.status_code == 200, submissions)
        filteredFailed = filter(lambda submission: submission.status_code != 200, submissions)
        for failedSubmission in filteredFailed:
            log.error("service=calendar\toper=mailhook\tstatus=failed_submission_to_mailhook\tstatus_code=%s\trequest=%s",
                      str(failedSubmission.status_code), str(failedSubmission.request.body))

        if len(filtered) == 0:
            raise CalendarError("calendar request failed for all uids")
