# -*- coding: utf-8 -*-
import os
import time
import logging
import socket

from kazoo.client import KazooClient
from kazoo.retry import KazooRetry
from kazoo.exceptions import ZookeeperError

from logbroker_client.consumers.simple.emitter import PartitionsCountEmitter
from logbroker_client.logbroker.client import LogbrokerMeta


log = logging.getLogger(__name__)


class DistributedEmitter(PartitionsCountEmitter):
    ZK_TIMEOUT = 10
    ZK_START_TIMEOUT = 60
    TIME_FOR_FULL_PARTY = 90
    TIME_FOR_FULL_LOCKED = 120

    def __init__(self, balancer_host, client, ident, workers_count, per_host_config, data_port=None):
        super(DistributedEmitter, self).__init__(
            balancer_host,
            client,
            ident,
            workers_count,
            per_host_config,
            data_port
        )
        if not self.host_config.get('zk_hosts'):
            raise ValueError('No zk_hosts parameter is stated')

    def build_lock_key(self):
        return '%s:%s' % (self.dc, ','.join(sorted(self.idents)))

    def get_own_party_members(self, zk_client, fqdn):
        party_path = '/partitions_queue/{client}/{key}/party/'.format(
            client=self.client,
            key=self.build_lock_key(),
        )
        try:
            return [member for member in zk_client.get_children(party_path)
                    if ('-%s-' % fqdn) in member]
        except ZookeeperError:
            return []

    def get_locks(self, zk_client):
        locks_path = '/partitions_queue/{client}/{key}/locks/'.format(
            client=self.client,
            key=self.build_lock_key(),
        )
        try:
            locks = zk_client.get_children(locks_path)
            locked = []
            for lock_name in locks:
                lock_nodes = zk_client.get_children(os.path.join(locks_path, lock_name))
                if lock_nodes:
                    locked.append(lock_name)
            return locked
        except ZookeeperError:
            return []

    def loop(self):
        client = LogbrokerMeta(self.balancer_host, self.client)
        log.info(
            'Load with config: balancer_host=%s client=%s dc=%s',
            self.balancer_host,
            self.client,
            self.dc,
        )
        hosts = client.hosts_info(self.dc)
        log.info('Hosts in dc=%s: %s', self.dc, hosts)
        log.info('Load topics for ident: %s', self.idents)
        partitions = []
        for ident in self.idents:
            ident_partitions = client.show_parts(ident=ident)
            partitions.extend(ident_partitions)

        filtered_topics = self.filter_topics(partitions)
        if not filtered_topics:
            log.warning('No topics to read from. Set correct `topics_filters` setting in config file!')
            self.INTERRUPTED = True
            return

        log.info('Topics: %s' % filtered_topics)
        filtered_partitions = self.filter_partitions(partitions, filtered_topics)
        log.info('Partitions for idents=%s: %s', self.idents, filtered_partitions)

        # create tasks to start workers
        for worker_index in xrange(self.workers_count):
            self.tasks_queue.put(
                {
                    'hosts': hosts,
                    'zk_hosts': self.host_config['zk_hosts'],
                    'client': self.client,
                    'lock_key': self.build_lock_key(),
                    'partitions': filtered_partitions,
                    'topics': filtered_topics,
                    'data_port': self.data_port
                }
            )

        zk_connection_retry = KazooRetry(max_tries=5, delay=0.5, backoff=2)
        zk_command_retry = KazooRetry(max_tries=5, delay=0.5, backoff=2)
        zk_client = KazooClient(
            hosts=self.host_config['zk_hosts'],
            connection_retry=zk_connection_retry,
            command_retry=zk_command_retry,
            timeout=self.ZK_TIMEOUT,
        )
        zk_client.start(timeout=self.ZK_START_TIMEOUT)

        start_wait_for_full_party = None
        start_wait_for_full_locked = None
        fqdn = socket.getfqdn()
        while not self.INTERRUPTED:
            time.sleep(5)
            members = self.get_own_party_members(zk_client, fqdn)
            # Проверяем, что все наши процессы участвуют в party
            if len(members) != self.workers_count:
                if start_wait_for_full_party is None:
                    start_wait_for_full_party = time.time()
                elif time.time() - start_wait_for_full_party > self.TIME_FOR_FULL_PARTY:
                    raise Exception('Not full party for a long time')
                log.warning('Not full party for %s seconds', int(time.time() - start_wait_for_full_party))
            else:
                start_wait_for_full_party = None

            # Проверяем, что все наши процессы хоть что-то делают
            locked = self.get_locks(zk_client)
            if len(locked) != len(filtered_partitions):
                if start_wait_for_full_locked is None:
                    start_wait_for_full_locked = time.time()
                elif time.time() - start_wait_for_full_locked > self.TIME_FOR_FULL_LOCKED:
                    raise Exception('Not full locked for a long time')
                log.warning('Not full locked for %s seconds', int(time.time() - start_wait_for_full_locked))
            else:
                start_wait_for_full_locked = None

        log.info('Emitter terminated')
