# -*- coding: utf-8 -*-
import time
import logging
import weakref

from logbroker_client.utils import importobj
from logbroker_client.handlers.base import BaseHandler
from logbroker_client.runner.workers.base import Worker
from logbroker_client.logbroker.client import (
    LogbrokerConsumer,
    BaseLogbrokerClientException,
)
from logbroker_client.handlers.exceptions import HandlerException
from ..exceptions import ExitException


log = logging.getLogger(__name__)
common_log = logging.getLogger('delta')


class SimpleHandlerDecorator(BaseHandler):
    def __init__(self, worker, handler):
        self.worker = worker
        self.handler = handler

    def process(self, header, data):
        if self.worker.INTERRUPTED:
            raise ExitException('Interrupted')
        return self.handler.process(header, data)

    def flush(self, force=False):
        return self.handler.flush(force)


class LogbrokerWorker(Worker):
    def __init__(self, handler):
        handler_cls = importobj(handler.get('class'))
        handler_args = handler.get('args')
        self.handler = SimpleHandlerDecorator(
            worker=weakref.proxy(self),
            handler=handler_cls(**handler_args),
        )

        self.INTERRUPTED = False
        super(LogbrokerWorker, self).__init__()

    def handle_sigint(self, signum, frame):
        self.INTERRUPTED = True

    def read_task(self):
        return self.tasks_queue.get()

    def create_client(self, task):
        return LogbrokerConsumer(
            task['hosts'],
            task['client'],
            task['topics'],
            task['partitions_count'],
            task['data_port'],
        )

    def loop(self):
        log.info('Run worker')
        try:
            task = self.read_task()
        except IOError:
            log.warning("Couldn't receive task from queue")
            return

        log.info('Got task %s', task)
        while not self.INTERRUPTED:
            client = self.create_client(task)
            try:
                client.read_unpacked(self.handler)
            except BaseLogbrokerClientException, e:
                log.warning('%s', e.__class__.__name__)
                time.sleep(0.1)
            except HandlerException, e:
                log.warning('%s: %s', e.__class__.__name__, e)
                time.sleep(0.1)
            except ExitException:
                log.info('Worker terminated')
                break


class AutoBalancedLogbrokerWorker(LogbrokerWorker):

    def create_client(self, task):
        return LogbrokerConsumer(
            task['hosts'],
            task['client'],
            task['topics'],
            task['partitions_count'],
            task['data_port'],
            suggest_timeout=120,
            suggest_balance='all',
        )
