# coding=utf-8
import StringIO
import logging

# Import regexen for smarter filename matching.
import os
import re
# Import time for measuring performance
import time

# version = 0.1.0

# Logbroker Consumer classes
# Main class which we inherit here
from logbroker_client.handlers.base import BaseHandler
# Class describing message and chunk structure
from logbroker_client.handlers.utils import MessageChunk
# Exceptions?
from logbroker_client.handlers.exceptions import HandlerException
# Import local stuff
import logbroker_client_common.utils as utils
from logbroker_client_common.unpacker import Unpacker
from logbroker_processors.utils.unistat import Unistat

log = logging.getLogger(__name__)


class CommonHandlerException(HandlerException):
    def __init__(self, message):
        log.debug("exception triggered: %s", message)


class CommonHandler(BaseHandler):
    TIMEOUT_ON_FLUSH_ERROR = 0.05

    def __init__(self, flush_lines, slow_log, format, stream):
        log.info("handler %s reinstantiated", self.__class__.__name__)
        self.props = {
            'format': format,
            'flush_lines': flush_lines,
            'slow_log': slow_log,
        }
        self.total_processed = 0
        self.chunk_received = 0
        self.processors = {}
        # instantiate line parsing class, aka format
        self.parser = utils.importobj(
            'logbroker_client_common.parsers.%s' % self.props.get('format', 'TSKV')
        )()
        # instantiate all processing classes
        for stream_type in stream.keys():
            module_path = 'logbroker_processors.%s' % (
                stream[stream_type].get('processor')
            )
            cls = utils.importobj(module_path)
            # Get filename
            filename = stream[stream_type].get('filename')
            # Get server mask
            server = stream[stream_type].get('server', '.*')
            # Instantiate class
            instance = cls(**stream[stream_type].get('args', {}))
            # Set stream propertis
            self.processors[stream_type] = {
                'filename': filename,
                'server': server,
                'filename_re': re.compile(filename),
                'server_re': re.compile(server),
                'inst': instance,
                'module_path': module_path,
                'lines_processed': 0
            }

    def process(self, header, chunk):
        self.chunk_received += 1
        matched_processors = []
        msg = MessageChunk(header, chunk)
        processing_begin = time.time()

        # header:
        # {'ident': 'mail_rpop',
        #  'offset': '3',
        #  'partition': '0',
        #  'path': '/tmp/test_logs/yrpop/access.log',
        #  'seqno': '9921',
        #  'server': 'logship-dev01e.cmail.yandex.net',
        #  'sourceid': 'base64:eL5wm2zzRya7j7ZHsKS2Zg',
        #  'topic': 'rt3.sas--mail_rpop--raw',
        #  'type': 'raw'}

        # data:
        # <blob> or text

        # the processing is done in "chunks".
        # each chunk consist of zero or more strings, separated with '\n'.

        for stream_type in self.processors.iterkeys():
            has_filename = self.processors[stream_type]['filename_re'].search(header.get('path', ''))
            has_server = self.processors[stream_type]['server_re'].search(header.get('server', ''))
            if has_filename and has_server:
                matched_processors.append(stream_type)

        for stream_type in matched_processors:
            current_processor = self.processors[stream_type]
            processor_instance = current_processor['inst']
            frames = processor_instance.split(chunk)
            for frame in frames:
                try:
                    parsed_line = self.parser.to_dict(frame)
                except ValueError:
                    # Skip bogus lines.
                    continue
                # Skip not parsed lines. Parser can skip lines by returning None.
                if parsed_line is None:
                    continue
                try:
                    # process the line.
                    if processor_instance.process(header, parsed_line):
                        current_processor['lines_processed'] += 1
                        self.total_processed += 1
                except ValueError as exc:
                    # wrong format. Skip this processor.
                    continue
                except Exception as exc:
                    if log.level == logging.DEBUG:
                        log.exception('processing exception: %s', exc)
                    else:
                        log.error('processing exception: %s', exc)

        # Check if limit of processed lines is reached.
        if self.total_processed >= self.props.get('flush_lines', 1000):
            log.debug('total lines processed: %i, calling flush()', self.total_processed)
            self.flush(force=True)

        # Count processing time
        total_time = time.time() - processing_begin
        if total_time >= float(self.props['slow_log']):
            log.warning('%s:%s processing time: %0.3f', msg.topic, msg.partition, total_time)

        # Required in commit call
        # https://github.yandex-team.ru/passport/logbroker-client/blob/master/logbroker-client/logbroker_client/logbroker/client.py#L283
        return True

    # flush is called occasionally, every Xth second.
    # We forcibly submit batch so that data wouldnt accumulate in case it is in low volume.
    def flush(self, force):
        flush_begin = time.time()
        # Try to flush each stream until it succeeds.
        # If the flush is successful, it is removed from the processing queue (streams)
        streams = (self.processors.keys())
        while streams:
            for stream_type in streams:
                try:
                    self.processors[stream_type]['inst'].flush(force)
                    log.debug(
                        '%s: lines processed: %i',
                        stream_type,
                        self.processors[stream_type]['lines_processed']
                    )
                    # Reset counter
                    self.processors[stream_type]['lines_processed'] = 0
                    # Remove this stream from the processing queue
                    streams.remove(stream_type)
                except Exception as exc:
                    if log.level == logging.DEBUG:
                        log.exception('%s: flush exception: %s', stream_type, exc)
                    else:
                        log.error('%s: flush exception: %s', stream_type, exc)

                    time.sleep(self.TIMEOUT_ON_FLUSH_ERROR)

        # Reset global processing counter.
        self.total_processed = 0

        # Count processing time
        total_time = time.time() - flush_begin
        if total_time >= float(self.props['slow_log']):
            log.warning('flush time: %0.3f', total_time)
