import struct

from google.protobuf.message import DecodeError

SYNC_WORD = '\x1F\xF7\xF7~\xBE\xA6^\2367\xA6\xF6.\xFE\xAEG\xA7\xB7n\xBF\xAF\x16\x9E\2377\xF6W\367f\xA7\6\xAF\xF7'


def _total_message_size(frame_size):
    return 4 + frame_size + len(SYNC_WORD)


def _try_parse_frame(data, pos):
    if len(data) > pos + 4:
        frame_size = struct.unpack('<I', data[pos:pos + 4])[0]
        total_size = _total_message_size(frame_size)
        end_pos = pos + total_size
        if len(data) >= end_pos:
            syncword_offset = pos + 4 + frame_size
            if data[syncword_offset:end_pos] == SYNC_WORD:
                frame = data[pos + 4:syncword_offset]
                return frame, total_size
    return None, 0


class Unpacker(object):
    """
    Class to process encoded data frames from given input byte-string
    """

    def __init__(self, data):
        self._data = data
        self._pos = 0

    def tail(self):
        return self._data[self._pos:]

    def exhausted(self):
        return self._pos >= len(self._data)

    def _fail_action(self, orig_pos):
        self._pos = len(self._data)
        return None, self._data[orig_pos:]

    def next_frame(self):
        """
        Gets next frame of data encoded using protoseq format
        :return: 2-tuple containing:
            1) str object with extracted data frame in case if next frame was found, else None
            2) str object with part of original data which was skipped, None if nothing was skipped
        """
        if self.exhausted():
            return None, None

        frame, data_read = _try_parse_frame(self._data, self._pos)
        if frame is not None:
            self._pos += data_read
            return frame, None

        orig_pos = self._pos
        while not self.exhausted():
            sw_pos = self._data.find(SYNC_WORD, self._pos)
            if sw_pos < 0:
                return self._fail_action(orig_pos)

            self._pos += sw_pos + len(SYNC_WORD)
            frame, data_read = _try_parse_frame(self._data, self._pos)
            if frame is not None:
                last_pos = self._pos
                self._pos += data_read
                return frame, self._data[orig_pos:last_pos]

        return self._fail_action(orig_pos)

    def next_frame_proto(self, message):
        """
        Gets next frame of data and sets it to given protobuf message
        if retrieved frame cannot be decoded as protobuf of given schema, skips this frame and tries to decode next and so on
        :param message: protobuf message to be set
        :return: 2-tuple containing:
            1) param message (filled with extracted data) if protobuf with required schema was found, else None
            2) str object with part of original data which was skipped, None if nothing was skipped
        """
        orig_pos = self._pos
        while not self.exhausted():
            frame, _ = self.next_frame()
            if frame is None:
                return self._fail_action(orig_pos)

            try:
                message.ParseFromString(frame)
                return message, self._data[orig_pos:self._pos - _total_message_size(len(frame))] or None
            except DecodeError:
                pass

        return self._fail_action(orig_pos)
