# encoding: utf-8
from __future__ import unicode_literals

import logging
from abc import ABCMeta, abstractmethod

from logbroker_client_common.unpacker import Unpacker
from logbroker_processors.processors import Processor
from logbroker_processors.utils.unistat import Unistat
from metrika.core.libs.appmetrica.protos.export import appmetrica_event_pb2
from .pusher import EventPushTokensPusher, AppMetricsPushError


class AppMetricsPTProcessor(Processor):
    """ Консьюмер для логов АппМетрики """
    _required_fields = frozenset((
        'DeviceID',
        'EventType',
        'AppPlatform',
        'AppID',
        'APIKey',
        'EventTimestamp',
    ))
    _other_fields = frozenset((
        'AccountID',
        'UUID',
        'EventValue',
        'EventName'
    ))

    def __init__(self, **args):
        super(AppMetricsPTProcessor, self).__init__()
        self._fields = self._required_fields | self._other_fields
        self.processors = [
            PushTokenRecordProcessor(**args),
            EventTypeCounter('EVENT_PROTOBUF_CRASH'),
            EventTypeCounter('EVENT_CRASH'),
            EventTypeCounter('EVENT_PROTOBUF_ERROR'),
            EventTypeCounter('EVENT_ERROR'),
            EventusCounter()
        ]

    def split(self, chunk):
        unpacker = Unpacker(chunk)
        while True:
            event, _ = unpacker.next_frame_proto(appmetrica_event_pb2.AppmetricaEvent())
            if event is None:
                return
            yield event

    def process(self, header, data):
        Unistat.increment('total_row_count_summ', 1)
        record = {k: v for k, v in data.iteritems() if k in self._fields}
        processed = False
        if self._validate(record):
            Unistat.increment('valid_row_count_summ', 1)
            for processor in self.processors:
                processed |= processor.process(record)
        if processed:
            Unistat.increment('processed_row_count_summ', 1)
        return processed

    def flush(self, force=False):
        pass

    def _validate(self, record):
        for f in self._required_fields:
            if f not in record:
                return False
        return True


class AppMetrikaRecordProcessor:
    __metaclass__ = ABCMeta

    @abstractmethod
    def process(self, record):
        """обработать и врнуть true, если это удалось"""


class PushTokenRecordProcessor(AppMetrikaRecordProcessor):
    _sandbox_accounts = frozenset((
        '103164068',
        '15020368',
        '18021017',
        '26691975',
        '316134018',
        '362849697',
        '38117510',
        '414697401',
        '420771248',
        '429725276',
        '441067589',
        '46871412',
        '554991636',
        '554993439',
        '86488496',
        '89241080',
    ))

    def __init__(self, **args):
        self._pusher = EventPushTokensPusher(**args.get('pusher', {}))
        self._sandbox_pusher = EventPushTokensPusher(
            **args.get('sandbox_pusher', {}))

    def process(self, record):
        event_type = record['EventType']
        if event_type != 'EVENT_PUSH_TOKEN' or not record.get('EventValue') or not record.get('UUID'):
            return False
        try:
            if self._is_sandbox_account(record):
                self._sandbox_pusher.push(record)
            else:
                self._pusher.push(record)
        except AppMetricsPushError as exc:
            logging.debug('Got error from pusher: %s', exc)
            return False
        return True

    def _is_sandbox_account(self, record):
        return 'AccountID' in record \
                and record['AccountID'] in self._sandbox_accounts


class EventTypeCounter(AppMetrikaRecordProcessor):
    def __init__(self, event_type):
        super(EventTypeCounter, self).__init__()
        self.event_type = event_type

    def process(self, record):
        event_type = record['EventType']
        if event_type != self.event_type:
            return False
        Unistat.increment('{}_{}_summ'.format(record['APIKey'], event_type), 1)
        Unistat.increment('{}_{}_{}_summ'.format(record['AppID'], record['AppPlatform'], event_type), 1)
        return True


class EventusCounter(AppMetrikaRecordProcessor):
    def __init__(self):
        super(EventusCounter, self).__init__()

    def process(self, record):
        if 'EventName' not in record:
            return False
        event_name = record['EventName']
        if not event_name.startswith('EVENTUS'):
            return False
        Unistat.increment('{}_EventName_{}_summ'.format(record['APIKey'], event_name), 1)
        Unistat.increment('{}_{}_EventName_{}_summ'.format(record['AppID'], record['AppPlatform'], event_name), 1)
        return True
