# encoding: utf-8
from __future__ import unicode_literals

import requests
import logging

from .utils import retry_on


class AppMetricsPushError(Exception):
    pass


class EventPushTokensPusher(object):
    """
    Отправка данных о событиях EVENT_PUSH_TOKEN
    """

    _platform_map = {
        'android': 'gcm',
        'iOS': 'apns',
    }

    def __init__(self, xiva_hub_url=None, allowed_apps=(), **kwargs):
        self._url_subscribe = '{}/subscribe_mobile'.format(xiva_hub_url)
        self._allowed_aps = set(allowed_apps)
        self._session = requests.Session()
        adapter = requests.adapters.HTTPAdapter(max_retries=3)
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

    def push(self, event):
        if event['AppID'] in self._allowed_aps:
            params, data = self._format_request(event)
            self._push_one_event(params, data)

    def _format_request(self, event):
        params = {
            'uid': event.get('AccountID', None) or event['DeviceID'],
            'uuid': event['UUID'],
            'platform': self._platform_map[event['AppPlatform']],
            'service': 'appmetrica_{}'.format(event['APIKey']),
            'origin': 'appmetrica',
            'ttl': 8760,
        }
        data = {
            'callback': 'xivamob:{}/{}'.format(event['AppID'], event['EventValue']),
        }
        return params, data

    @retry_on(exc=(AppMetricsPushError,))
    def _push_one_event(self, params, data):
        try:
            r = self._session.post(self._url_subscribe, params=params, data=data)
            r.raise_for_status()

            logging.debug('Event pushed to xiva with params: %s', params)
        except requests.RequestException:
            raise AppMetricsPushError('Can not push to xiva. Params: %s' % params)
