import logging
import socket
import time
from datetime import datetime
from os import environ
from pprint import pprint

import logbroker_client_common.utils as utils

from ..processors import Processor

log = logging.getLogger(__name__)


class MockGraphiteBackend(object):
    def send(self, data):
        pprint(data)


class FileGraphiteBackend(object):
    def __init__(self, path='/tmp/consumer_graphite.log'):
        self.file = open(path, 'w+')

    def send(self, data):
        for line in data:
            self.file.write(line + '\n')
        self.file.flush()


class GraphiteBackend(object):
    def __init__(self, **opts):
        self.host = opts.get('host', 'localhost')
        self.port = opts.get('port', '42000')
        self.timeout = opts.get('timeout', 10)
        self.attempts = opts.get('attempts', 3)

    def send(self, data):
        if type(data) is str:
            data = [data]
        if type(data) is not list:
            raise TypeError('data argument must be either list or a str')
        # dont bother with wrong or empty batch.
        if not data:
            return True

        log.info('sending {l} lines of metric data, server:{s}, port:{p}'.format(l=len(data), s=self.host, p=self.port))

        for x in xrange(self.attempts):
            try:
                self.__do_send(data)
                return True
            except Exception as e:
                log.exception(e)
                time.sleep(1)
                continue

    def __do_send(self, data):
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(self.timeout)
            sock.connect((self.host, int(self.port)))
            for metric in data:
                sock.send(metric + '\n')
        finally:
            sock.close()


class GraphiteMetrics(object):
    def __init__(self, backlog=600):
        # metric = {
        #         'mdb.userjournal.mdb320a.yserver_imap.authorization': {
        #             '1444121516': '987',
        #             '1444121844': '123'
        #             }
        #         ...: { ... }
        self.metric = {}
        self.backlog = backlog

    def add(self, metric, time, value):
        return self.__update(metric, time, value, op='add')

    def set(self, metric, time, value):
        """
        Set metric to the new value.
        """
        return self.__update(metric, time, value, op='set')

    def increment(self, metric, time, value=1):
        """
        Add a fixed value to the given metric.
        """
        return self.__update(metric, time, value, op='add')

    def __update(self, metric, time, value, op):
        # First, check if we still accept data for this period.
        if not self.__validate_ts(time):
            return False
        # Then, round ts to the nearest minute.
        ts = self.__nearest_minute(time)
        # Make sure this path exists, and if not -- create.
        self.__ensure_metric_ts(metric, ts)
        # Finally, update the metric.
        if op == 'set':
            self.metric[metric][ts] = value
        elif op == 'add':
            self.metric[metric][ts] += value
        return True

    def dump(self):
        """
        Returns all metrics that are older than backlog period.
        """
        dump = []
        # For testing purposes, we need to predefine now()
        now = int(environ.get("__CURRENT_TIMESTAMP", time.time()))
        for name, periods in self.metric.items():
            for minute, value in periods.items():
                # Skip the counter if it is not too old.
                if self.__nearest_minute(now) - minute < self.backlog:
                    # log.info('{} ts is too early for dump, {} < {}'.format(minute, self.__nearest_minute(now) - minute, self.backlog))
                    continue
                dump.append('{metric} {val} {ts}'.format(metric=name, val=value, ts=minute))
                del(periods[minute])
        return dump

    def dump_all(self):
        """
        Returns currently accumulated metrics as a list.
        """
        return ['{metric} {val} {ts}'.format(metric=name, val=val, ts=ts)
                for name, periods in self.metric.items()
                for ts, val in periods.items()]

    def __validate_ts(self, ts):
        """
        Check if the given timestamp falls within the predefined period.
        Return false if it does not.
        """
        # For testing purposes, we need to predefine now()
        now = environ.get("__CURRENT_TIMESTAMP", time.time())

        if self.__nearest_minute(int(now)) - self.__nearest_minute(ts) <= self.backlog:
            return True
        else:
            # log.info('{} old timestamp, skipping: {} < {}'.format(ts, self.__nearest_minute(int(now))- self.__nearest_minute(ts), self.backlog))
            return False

    def __nearest_minute(self, ts):
        """
        Return unix TS rounded to the nearest minute.
        """
        rounded = datetime.fromtimestamp(int(ts)).replace(second=0)
        return int(rounded.strftime('%s'))

    def __ensure_metric_ts(self, metric, ts):
        """
        Ensure that path to metric and ts does exist.
        """
        if metric not in self.metric:
            self.metric[metric] = {}
        if ts not in self.metric[metric]:
            self.metric[metric][ts] = 0


class BaseGraphiteProcessor(Processor):
    def __init__(self, **opts):
        # Accumulator object
        self.metrics = GraphiteMetrics(backlog=opts.get('backlog_depth', 600))

        # Graphite sender.
        backend = globals()[opts.get('backend', 'GraphiteBackend')]
        self.graphite = backend(**opts.get('backend_opts', {}))

        # Announce to the logs -- this marks a restart.
        log.warning("Graphite consumer {} process started".format(self.__class__.__name__))

        self.extractors = self.__get_extractors()

        # Hostname.
        self.hostname = socket.getfqdn().replace('.', '_')

    def __get_extractors(self):
        extractors = []
        for cmd in dir(self):
            conditions = [
                # Only functions
                hasattr(getattr(self, cmd), '__call__'),
                # Not internal functions
                not cmd.startswith('_'),
                # Not any external methods related
                # to the base functionality
                cmd not in {'flush', 'process'}
            ]
            if all(conditions) is True:
                extractors.append(getattr(self, cmd))
        log.info('registered following data extractors: {}'.format(extractors))
        return extractors

    def process(self, header, data):
        for func in self.extractors:
            try:
                func(data, header)
            except Exception as e:
                log.error('exception at extractor func "{f}": {e}, meta: {h}, data: {d}'.format(f=func.__name__, e=e, d=data, h=header))

        return True

    def flush(self, force=True):
        self.graphite.send(self.metrics.dump())
