import time
from random import random
import logging
import storage
import hashlib
import re
from os import environ as env

log = logging.getLogger(__name__)


class BaseProcessor(object):

    def __init__(self, **args):
        self.date_format = '%Y-%m-%d %H:%M:%S'
        log.warning("%s processor initialized." % (self.__class__.__name__))
        self.args = args
        # Can be Hbase and MockBase for testing
        self.db = getattr(storage, args.get('type', 'Hbase'))(**args)

    def __put_cf(self, source_dict, cf='cf'):
        if not isinstance(source_dict, dict):
            raise TypeError("%s.prepend_cf(): argument 1 must be a dict" % __name__)
        if not source_dict:
            raise ValueError("Data dictionary must not be empty")
        new_dict = {}
        for key, value in source_dict.iteritems():
            new_dict['%s:%s' % (cf, key)] = value
        return new_dict

    def reverse_ts(self, ts, msecs=None, base=(2**64)):
        ms = msecs
        if ms is None:
            # Set it 'RANDOM_NUM' environment variable (for tests)
            # Or a random number (0.0 .. 1.0)
            ms = float(env.get('RANDOM_NUM', random()))

        if isinstance(ts, time.struct_time):
            ts_float = time.mktime(ts) + float(ms)
        elif isinstance(ts, float):
            ts_float = ts
        elif isinstance(ts, int):
            ts_float = ts + ms
        else:
            raise TypeError("%s.reverse_ts(): argument 1 must be float, int, or time.time_struct!" % __name__)
        return (base - int((ts_float) * 1000))

    def md5(self, string):
        return hashlib.md5(string).hexdigest()

    def filter_dict(self, input_dict, keys_to_remove):
        if not isinstance(keys_to_remove, list):
            raise TypeError("%s.filter_dict(): second argument must be a list!" % __name__)
        keep_keys = set(input_dict.keys()) - set(keys_to_remove)
        return {key: input_dict[key] for key in keep_keys}

    def process(self, header, data):
        """
        This method looks for a matching filename in
        'events' dictionary, and finding a match,
        fires the data processing method (see getattr()).
        The returned result is fed into Hbase.accumulate() instance,
        which later will be processed in self.flush().
        """

        for method_name, opts in self.args['events'].iteritems():
            file_match = re.search(opts.get('filename', '.+'), header.get('path', 'unknown'))
            source_host_match = re.search(opts.get('server', '.+'), header.get('server', ''))
            if file_match and source_host_match:
                try:
                    # Format the data and form a key.
                    event_data = getattr(self, method_name)(header, data)
                    # event_data = {
                    #     'data': data,
                    #     'cf': self.cf,
                    #     'table': self.table,
                    #     'key': key
                    # }
                    self.db.accumulate(
                        table=event_data.get('table', method_name),
                        key=event_data['key'],
                        data=self.__put_cf(event_data['data'], event_data.get('cf', 'cf'))
                        )

                except ValueError:
                    # log.debug('value error: %s' % (e))
                    # wrong format. Skip this method_name.
                    continue
                except KeyError as e:
                    log.error('parser "{p}" did not return a required attribute: {a}'.format(
                        p=method_name,
                        a=unicode(e)
                        )
                    )
                except Exception as e:
                    log.error('parser "{p}" general error: {e}'.format(
                        p=method_name,
                        e=unicode(e)
                        )
                    )
        return True

    def flush(self, force):
        try:
            self.db.commit()
        except Exception as e:
            log.error("flush - unhandled exception at sending batch: %s", unicode(e))
