# encoding: utf-8
from __future__ import unicode_literals

from collections import namedtuple
from functools import partial

from .records import (BaseUserJournalRecord, ReceiveRecord, AbuseRecord,
                      MarkRecord, MoveRecord, StartReadingRecord, StopReadingRecord,
                      ViewRecord, LinkClickRecord)

OperationConfig = namedtuple('OperationConfig', ['record_class', 'mid_mode'])


class UserJournalParser(object):
    OPERATIONS = {
        'receive': OperationConfig(ReceiveRecord, 'single'),
        'abuse': OperationConfig(AbuseRecord, 'multi'),
        'mark': OperationConfig(MarkRecord, 'multi'),
        'trash': OperationConfig(BaseUserJournalRecord, 'multi'),
        'delete': OperationConfig(BaseUserJournalRecord, 'multi'),
        'move': OperationConfig(MoveRecord, 'multi'),
        'startReading': OperationConfig(StartReadingRecord, 'single'),
        'stopReading': OperationConfig(StopReadingRecord, 'single'),
        'view': OperationConfig(ViewRecord, 'single'),
        'linkClick': OperationConfig(LinkClickRecord, 'single'),
    }

    def _parse_single_mid(self, cfg, data):
        return [cfg.record_class(**data)]

    def _parse_multi_mid(self, cfg, data):
        mids = data.get('mids', '').split(',')
        return self._make_mids_records(cfg, data, mids)

    def _make_mids_records(self, cfg, data, mids):
        if not mids:
            return None
        if len(mids) == 1:
            return [cfg.record_class(mid=mids[0], **data)]
        partial_construct = partial(cfg.record_class, **data)
        return [partial_construct(mid=mid) for mid in mids]

    def parse(self, data):
        if not data:
            return None
        operation = data.get('operation')
        if not operation:
            return None
        cfg = self.OPERATIONS.get(operation)
        if not cfg:
            return None

        if cfg.mid_mode == 'multi':
            return self._parse_multi_mid(cfg, data)
        elif cfg.mid_mode == 'single':
            return self._parse_single_mid(cfg, data)
        else:
            return None
