# encoding: utf-8
from __future__ import unicode_literals

import logging

from collections import deque
from time import sleep

from logbroker_processors.processors import Processor
from .parsers import SendrDeliveryParser, SendrClickParser
from logbroker_processors.utils import ClickhouseTSKVPusher

loger = logging.getLogger(__name__)


class BaseProcessor(Processor):
    """Базовый класс процессоров статистики Рассылятора"""
    _parser_cls = None
    _tskv_format = None

    def __init__(self, *args, **kwargs):
        super(BaseProcessor, self).__init__(*args, **kwargs)
        self._parser = self._parser_cls()
        self._buffer = deque()
        self._pushers = [ClickhouseTSKVPusher(loger, **opts) for opts in kwargs['clickhouse']]

    def process(self, header, data):
        if data.get('tskv_format') != self._tskv_format:
            return False
        record = self._parser.parse(data)
        if not record:
            return False
        if not record.valid:
            return False
        self._buffer.append(record)
        return True

    def _flush_pause(self):
        # Hack for enlarge clickhouse insert batch size
        # Fixes DB::Exception: Too much parts. Merges are processing significantly slower than inserts.
        sleep(10)

    def flush(self, force=False):
        loger.debug('Flushing %d records to database', len(self._buffer))
        for pusher in self._pushers:
            pusher.push(self._buffer)
        self._buffer.clear()
        self._flush_pause()


class DeliveryProcessor(BaseProcessor):
    """Процессор записей sendr-delivery-log"""
    _parser_cls = SendrDeliveryParser
    _tskv_format = 'sendr-delivery-log'


class ClickProcessor(BaseProcessor):
    """Процессор записей sendr-click-log"""
    _parser_cls = SendrClickParser
    _tskv_format = 'sendr-click-log'
