# encoding: utf-8
from __future__ import unicode_literals
import time
import logging
from datetime import datetime
from blackboxer import Blackbox, environment
from blackboxer.exceptions import BlackboxError

from logbroker_client_common.utils import chunk_it
from logbroker_processors.processors import Processor as BaseProcessor

from ..utils import passport

from .base import BaseAPIError, DataParser, ParseError, MetaMailAPI, MetaMailError, SoAPI

logger = logging.getLogger(__name__)


class MailLogParser(DataParser):
    DATE_FORMAT = '%Y-%m-%d %H:%M:%S'

    required_fields = [
        'timestamp',
        'client',
        'folder',
        'ip',
        'mid',
        'seen',
        'suid',
        'type',
        'uid',
        'stid',
        'received_date'
    ]

    exclude_fields = ['tskv_format', 'timezone', 'source']

    def __init__(self, *args, **kwargs):
        super(MailLogParser, self).__init__(*args, **kwargs)

        self.blackbox = Blackbox(url=environment.url_from_yenv(yenv_name='other', yenv_type='production'))
        self.blackbox_corp = Blackbox(url=environment.url_from_yenv(yenv_name='intranet', yenv_type='production'))

        self.meta_mail = MetaMailAPI(host=self._cfg['meta']['host'],
                                     port=self._cfg['meta']['port'],
                                     path=self._cfg['meta']['path'])

        self.meta_mail_corp = MetaMailAPI(host=self._cfg['meta']['host_corp'],
                                          port=self._cfg['meta']['port'],
                                          path=self._cfg['meta']['path'])

    @property
    def source(self):
        if self._raw_data:
            return self._raw_data.get('source', 'wmilog')

        return 'wmilog'

    def clean(self):
        data = super(MailLogParser, self).clean()
        try:
            timestamp = data.pop('timestamp')
            received_date = data.pop('received_date')

            data['data'] = int(received_date)
            try:
                data['unixtime'] = int(timestamp)
            except ValueError:
                data['unixtime'] = int(time.mktime(datetime.strptime(timestamp, str(self.DATE_FORMAT)).timetuple()))

            uid = int(data['uid'])
            if passport.user_is_corp(uid):
                # Если team пользователь
                info = self.blackbox_corp.userinfo(userip='127.0.0.1', uid=uid)['users'][0]
            else:
                info = self.blackbox.userinfo(userip='127.0.0.1', uid=uid)['users'][0]

            data.update({
                'login': info['login'],
                'karma': info['karma']['value'],
                'karma_status': info['karma_status']['value']
            })

        except (ValueError, KeyError) as e:
            logging.warning('[DataError] %s', e.message)
            raise ParseError
        except BlackboxError as e:
            logger.warning('[BlackBoxError] %s ', e.message)
            raise ParseError
        except MetaMailError as e:
            logger.warning('[MetaMailError] %s', e.message)
            raise ParseError

        return data


class IMAPLogParser(MailLogParser):
    """
    У IMAP чуть другой формат логов, делаем отдельный парсер и процессор
    """
    source = 'imaplog'
    exclude_fields = ['tskv_format', 'timezone', 'session', 'source']
    DATE_FORMAT = '%Y-%m-%d %H:%M:%S.%f'


class FastSrvLogParser(MailLogParser):
    """
    У FastSrv чуть другой формат логов, делаем отдельный парсер и процессор
    """
    source = 'fastsrvlog'
    required_fields = [
        'timestamp',
        'folder',
        'ip',
        'mid',
        'seen',
        'suid',
        'type',
        'uid'
    ]
    exclude_fields = ['tskv_format', 'timezone', 'session', 'source']


class MailSOProcessor(BaseProcessor):
    log_parser_cls = MailLogParser

    def __init__(self, **opts):
        self.cfg = opts
        self.so_api = SoAPI(host=self.cfg['so']['host'],
                            port=self.cfg['so']['port'],
                            path=self.cfg['so']['path'])

        self.parser = self.log_parser_cls(self.cfg)
        self._reports = {}

        super(MailSOProcessor, self).__init__()

    def _get_reports(self):
        for source, source_reports in self._reports.iteritems():
            for report_type, reports in source_reports.iteritems():
                for chunk in chunk_it(reports, size=50):
                    yield source, report_type, chunk

    def _clear_reports(self, source, report_type):
        if source in self._reports and report_type in self._reports[source]:
            self._reports[source][report_type] = []

    def process(self, header, data):
        if self.parser.is_valid(data):
            report = self.parser.cleaned_data
            report_type = report.pop('type')

            source_reports = self._reports.setdefault(self.parser.source, {})
            type_reports = source_reports.setdefault(report_type, [])
            type_reports.append(report)

        return True

    def flush(self, force=False):
        for source, report_type, reports in self._get_reports():
            try:
                self.so_api.report(reports, report_type, source=source)
            except BaseAPIError as e:
                logger.error('[SOError] %s', e.message)
            finally:
                self._clear_reports(source, report_type)


class ImapSOProcessor(MailSOProcessor):
    log_parser_cls = IMAPLogParser


class FastSrvSOProcessor(MailSOProcessor):
    log_parser_cls = FastSrvLogParser
