# encoding: utf-8
from __future__ import unicode_literals

import time
import random
import requests
from os import environ

environ['REQUESTS_CA_BUNDLE'] = '/etc/ssl/certs/ca-certificates.crt'


class ClickhouseTSVPusher(object):

    def __init__(self, logger, **opts):
        self.logger = logger
        self._opts = opts
        self._timeout = self._opts.get('timeout', 5)
        self._port = self._opts.get('port', '8443')
        self._choose_host()
        self._log_data_errors = opts.get('log_data_errors', False)

    def _form_query(self, table, columns):
        request_string = 'INSERT INTO {db}.{t} ({f}) FORMAT TabSeparated'
        return request_string.format(
            db=self._opts['database'],
            t=table,
            f=', '.join(columns)
        )

    def _choose_host(self):
        host = random.choice(self._opts['hosts'])
        self.logger.debug('clickhouse host: %s', host)
        if self._opts.get('user'):
            self._url = 'https://{user}:{password}@{host}:{port}/'.format(
                user=self._opts['user'],
                password=environ[self._opts['password_env']],
                port=self._port,
                host=host)
        else:
            self._url = 'https://{host}:{port}/'.format(
                host=host,
                port=self._port
            )

    def _form_record(self, columns, record):
        line = []
        appender = line.append
        for column in columns:
            appender(
                unicode(record[column]).replace("\\", "\\\\")
            )
        return '\t'.join(line)

    def _try_push(self, table, columns, data):
        try:
            req = requests.post(
                self._url,
                params={
                    'query': self._form_query(table, columns)
                },
                data=data,
                timeout=self._timeout,
            )
            if req.status_code == requests.codes.ok:
                return True
            self.logger.error('remote error: %s', req.text)
        except requests.exceptions.RequestException as exc:
            self.logger.exception(exc)
            # Facilitate cross-host retries
            self._choose_host()
            time.sleep(1)
        if self._log_data_errors:
            self.logger.debug('columns: %s', str(columns))
            self.logger.debug('data: %s', str(data))
        # wait to avoid crapping up logs on formatting skew errors.
        time.sleep(1)
        return False

    def push(self, table, columns, records):
        if not records:
            return
        tsv_data = []
        record_num = 0
        for record in records:
            tsv_data.append(
                self._form_record(columns, record)
            )
            record_num += 1
        push_data = '\n'.join(tsv_data).encode('utf-8')
        while True:
            pushed = self._try_push(
                table, columns, data=push_data
            )
            if pushed:
                break
        self.logger.debug(
            '%d records written',
            record_num,
        )
