import urllib2
import psycopg2
import base64

import magic
import chardet
import json

from psycopg2 import InterfaceError, OperationalError
from collections import namedtuple
from retry import retry

from .mdb import get_st_id, mdb_put, get_offsets
from .tools import safe_unicode, http_get
from .windat_logger import get_logger

log = get_logger(__name__)


class WindatId(namedtuple('WindatId', ['uid', 'mid', 'hid'])):
    def __repr__(self):
        return 'uid={} mid={} hid={}'.format(*self)


class LoggingCursor(psycopg2.extensions.cursor):
    def execute(self, sql, args=None):
        try:
            psycopg2.extensions.cursor.execute(self, sql, args)
        except:
            log.warning(self.mogrify(sql, args))
            raise


MAX_CHARS_SAMPLE = 60


def get_pg_dsn(uid, host, timeout):
    get_url = 'http://{}/conninfo?uid={}&mode=master&format=json'.format(host, uid)
    result = json.loads(http_get(get_url, timeout=timeout))
    addr = result['addrs'][0]
    dsn = 'host={host} port={port} dbname={dbname}'.format(**addr)
    return dsn


def prepare_mime(mime_dict):
    mime_part_headers = [
        'hid', 'content_type', 'content_subtype', 'boundary',
        'name', 'charset', 'encoding', 'content_disposition',
        'filename', 'cid', 'offset_begin', 'offset_end'
    ]

    return tuple(mime_dict.get(k, None) for k in mime_part_headers)


def prepare_params(filename, raw, uid, mid, hid):
    type, subtype = magic.from_buffer(raw, mime=True).split('/')
    encoded = base64.encodestring(raw)
    encoding = chardet.detect(raw[:MAX_CHARS_SAMPLE])['encoding']

    offset_begin = 0
    offset_end = len(encoded)

    mime = {
        'hid': hid,
        'content_type': type,
        'content_subtype': subtype,
        'name': safe_unicode(filename, 'name'),
        'charset': encoding,
        'encoding': 'base64',
        'offset_begin': offset_begin,
        'offset_end': offset_end
    }

    return {
        'uid': uid,
        'mid': mid,
        'hid': hid,
        'windat': prepare_mime(mime)
    }


def process_attachment(cursor, params, encoded_attach, mulca_client):
    log.debug('mulca_put')
    st_id = mulca_client.mulca_put(encoded_attach)
    log.debug('mulca_put ok')
    params.update({'st_id': st_id})
    log.debug('mdb_put')
    mdb_put(cursor, params)
    log.debug('Successfully inserted mdb_put')


@retry((InterfaceError, OperationalError), tries=10, delay=1, jitter=(1, 2), logger=log)
def process_windat(uid, mid, hid, config, mulca_client=None, tnef_parser=None):
    timeout = int(config['timeout'])
    conn_string = 'user=%s ' % config['user'] + get_pg_dsn(uid, config['sharpei'], timeout)
    with psycopg2.connect(conn_string) as conn:
        log.debug('gotta conn: %s' % conn)
        conn.set_session(autocommit=True)
        cursor = conn.cursor(cursor_factory=LoggingCursor)
        st_id = get_st_id(cursor, uid, mid)
        if not st_id:
            log.info('failed to get st_id')
            return

        offset_begin, offset_end = get_offsets(cursor, uid, mid, hid)
        if not offset_begin:
            log.warning('failed to get offsets for st_id=%s' % st_id)
            return

        log.debug('Getting winmail.dat..')
        windat_b64_encoded = mulca_client.get_attach(st_id, offset_begin, offset_end)
        log.debug('Decoding winmail.dat')
        windat = base64.b64decode(windat_b64_encoded)
        log.debug('Parsing winmail.dat')
        try:
            attachments = tnef_parser.parse(windat)
        except Exception as e:
            log.warning(str(e))
            return

        if len(attachments) == 0:
            log.info('failed to parse st_id=%s' % st_id)
            return

        log.debug('successfully parsed st_id=%s count=%s' % (st_id, len(attachments)))
        for i, attach in enumerate(attachments):
            filename, raw = attach.getName(), attach.getData()
            params = prepare_params(filename, raw, uid, mid, hid + '.' + str(i + 1))
            encoded = base64.encodestring(raw)
            if encoded:
                log.debug('processing attachment: #%s %s' % (i, filename))
                process_attachment(cursor, params, encoded, mulca_client)
                log.debug('successfully processed #%s' % i)
            else:
                log.info('attachment #%s %s is empty, skip' % (i, filename))
        log.debug('successfully processed winmail.dat')
