# encoding: utf-8
import logging
from threading import RLock

import six
import base64
import requests
from time import time
from requests.exceptions import ConnectionError, Timeout, RequestException, HTTPError


from .v2 import tvm2
from .common.exceptions import TVMNetworkError, TVMResponseError, TicketParsingException
from .common.blackbox import get_blackbox_params

TVM_KEYS_REFRESH_INTERVAL = 23 * 3600  # 23 часа


logger = logging.getLogger('utvm')


class TVM(object):
    """
    Класс хранит настройки получения и проверки TVM тикетов.
    На данный момент содержит работу с получением сервисных тикетов и
    проверки сервисных и пользовательских тикетов.

    Абстрагируются получение и проверка тикетов TVM версии 1 и 2, но за это
    ограничевается только совместимыми методами получения тикетов
    """

    _keys_loaded = None
    _tvm2_keys = None

    def __init__(
        self, client_id, secret,
        tvm_host='tvm-api.yandex.net',
        local_v1_keys=None,
        allowed_v1_clients=None,
        passport_env=None,
        use_v1=False,
        update_keys_interval=TVM_KEYS_REFRESH_INTERVAL,
    ):
        """
            :param client_id: идентификатор клиента, для которого запрашивать тикеты
            :param secret: секрет приложения в base64 кодировке
            :param tvm_host: хост для получения ключей и тикетов. По умолчанию 'tvm-api.yandex.net'
            :param local_v1_keys: ключи TVM 1 загруженные оффлайн. (deprecated)
            :param allowed_v1_clients: клиенты TVM 1 для которых надо плучить ключи (deprecated)
            :param passport_env: окружение ЧЯ для выписывания токенов похода за пользовательскими тикетами
            :param use_v1: можно выключить получение старых тикетов (deprecated)
        """

        self.client_id = client_id
        self.tvm_host = tvm_host
        self.test_v2_ticket = None

        if isinstance(secret, six.text_type):
            secret = secret.encode('ascii')
        self._secret = secret
        self._decoded_secret = self._decode_secret(secret)

        self.use_passport = bool(passport_env)
        if self.use_passport:
            self.passport_env = passport_env
            self.passport_host, self.passport_client = get_blackbox_params(
                passport_env)

        self._update_keys_lock = RLock()
        self._update_keys_interval = update_keys_interval
        self.update_public_keys()

    def ticket_getter(self):
        return TicketGetter(self)

    def ticket_checker(self):
        return TicketChecker(self)

    @property
    def secret_v2(self):
        """TVM 2.0 formatted client secret"""
        return self._secret

    @property
    def tvm2_keys(self):
        """Cached TVM 2.0 public keys"""
        return self._tvm2_keys

    @property
    def keys_loaded_at(self):
        """When public keys where updated"""
        return self._keys_loaded

    def force_tvm2_keys(self, data):
        """
        Используется для тестирования вместе с ticket-parser2-unittest
        Задает публичный ключ для проверки тикетов в константу
        """
        self._tvm2_keys = data
        self._keys_loaded = time() * 2

    def force_tvm2_ticket(self, dst, ticket):
        """
        Используется для тестирования вместе с ticket_parser2-unittest
        Задает токен для запросов к системе с тестовым публичным ключем
        """
        self.test_v2_ticket = ticket

    @staticmethod
    def _decode_secret(secret):
        """Decode base64 client secret"""
        padded_secret = secret + b'=' * (len(secret) % 4)
        return base64.urlsafe_b64decode(padded_secret)

    def _load_v2_keys(self):
        """Load public TVM 2.0 keys"""
        url = 'https://{host}/2/keys/'.format(host=self.tvm_host)
        try:
            response = requests.get(
                url,
                params={'lib_version': tvm2.version()},
                verify='/etc/ssl/certs/ca-certificates.crt'
            )
            response.raise_for_status()
            return response.content
        except ConnectionError:
            raise TVMNetworkError
        except Timeout:
            raise TVMResponseError
        except HTTPError:
            raise TVMResponseError
        except RequestException:
            raise TVMResponseError

    def _need_to_update_keys(self):
        if self._tvm2_keys and self._keys_loaded:
            return (time() - self._keys_loaded) >= self._update_keys_interval
        else:
            return True

    def update_public_keys(self):
        # don't want to take lock each time condition is checked - take lock only for positive case
        # and then re-check condition under lock
        if not self._need_to_update_keys():
            return False
        else:
            with self._update_keys_lock:
                if not self._need_to_update_keys():
                    return False
                preload_time = time()
                self._tvm2_keys = self._load_v2_keys()
                self._keys_loaded = preload_time
                logger.debug('Public TVM2 keys are updated at %s', preload_time)
                return True


class _RefreshContextMixin(object):
    def __init__(self, tvm_interface):
        self._tvm = tvm_interface
        self._init_contexts()
        self._keys_loaded_at = tvm_interface.keys_loaded_at

    def _init_contexts(self):
        raise NotImplementedError

    def _refresh_context(self):
        """Recreate context objects if public keys where updated."""
        # tvm instance maybe shared between checkers / getters
        self._tvm.update_public_keys()
        if self._keys_loaded_at < self._tvm.keys_loaded_at:
            self._init_contexts()
            self._keys_loaded_at = self._tvm.keys_loaded_at


class TicketGetter(_RefreshContextMixin):
    """
        Класс для получения сервисных ключей обоих версий.
        Ключи TVM1 получаются через интерфейс TVM2.
    """

    def get_service_ticket_headers(self, destinations, scopes=None):
        """
            Получение тикетов в формате пригодном для использования в заголовках запросов
            :param  destinations: список приложений TVM в которые собираемся ходить.
            :param  scopes: скоупы для тикета TVM2.
            :returns: dict формата {'<destination>':{
                    'X-Ya-Service-Ticket': 'TVM2 ticket',
                    'Ticket': 'TVM1 ticket',
                    'TicketClId': 'client_id',
                }}
        """
        self._refresh_context()

        destinations = self._fix_list(destinations)

        if self._tvm.test_v2_ticket:
            # Для TVM задан тестовый тикет. Возвращаем только его заголовки
            # Ограничение - сейчас не производится проверка цели для тикета
            headers_by_destinations = {}
            for dst in destinations:
                headers_by_destinations[str(dst)] = {'X-Ya-Service-Ticket': self._tvm.test_v2_ticket}
            return headers_by_destinations

        ts = int(time())
        ts_bytes = b'%d' % ts

        str_dest = b','.join((self._force_bytes(x) for x in destinations))

        params = {
            'grant_type': 'client_credentials',
            'src': self._tvm.client_id,
            'dst': str_dest,
            'ts': ts_bytes,
            'scopes': scopes,
            'sign': self._service_context.sign(ts_bytes, destinations, scopes=scopes),
        }

        response = self._request_service_tickets(params)

        depr_headers = {}

        headers_by_destinations = {}
        for dst in destinations:
            headers = {'X-Ya-Service-Ticket': response[str(dst)]['ticket']}
            headers.update(depr_headers)
            headers_by_destinations[str(dst)] = headers

        return headers_by_destinations

    def get_passport_ticket(self):
        """ Возвращает набор хедеров для похода в ЧЯ за тикетами """
        if not self._tvm.use_passport:
            return {}
        passport_client_id = tvm2.BlackboxClientId[self._tvm.passport_client].value
        headers = self.get_service_ticket_headers(passport_client_id)
        return headers[str(self._tvm.passport_client)]

    def _init_contexts(self):
        """ Создание контекста для подписи TVM2 """
        self._service_context = tvm2.ServiceContext(
            self._tvm.client_id,
            self._tvm.secret_v2,
            self._tvm.tvm2_keys
        )
        self._keys_loaded_at = self._tvm.keys_loaded_at

    @staticmethod
    def _fix_list(value):
        """ Если нужен именно list """
        if isinstance(value, list):
            return value
        elif isinstance(value, (set, tuple)):
            return list(value)
        else:
            return [value]

    @staticmethod
    def _force_bytes(value):
        """ Когда хочется bytes из всего """
        if isinstance(value, six.binary_type):
            return value
        if isinstance(value, int):
            value = six.text_type(value)
        if isinstance(value, six.text_type):
            return value.encode('utf-8')
        raise TypeError

    def _request_service_tickets(self, params):
        """ Бегаем за тикетом в ручку TVM """
        url = 'https://{host}/2/ticket/'.format(host=self._tvm.tvm_host)
        try:
            response = requests.post(
                url,
                data=params,
                verify='/etc/ssl/certs/ca-certificates.crt'
            )
            response.raise_for_status()
            return response.json()
        except ConnectionError:
            raise TVMNetworkError
        except Timeout:
            raise TVMResponseError
        except HTTPError:
            raise TVMResponseError
        except RequestException:
            raise TVMResponseError


class TicketChecker(_RefreshContextMixin):
    """Check TVM tickets."""

    def check_tickets(self,
                      tvm_2_service_ticket=None,
                      tvm_2_user_ticket=None):
        """
        Same as check_headers but tickets can be passed explicitly as arguments.
        Returns:
            TicketCheckResult
        """
        self._refresh_context()

        service_ticket = None
        user_ticket = None

        if tvm_2_service_ticket:
            try:
                service_ticket = self._service_context.check(tvm_2_service_ticket)
            except TicketParsingException as e:
                logger.debug('Bad TVM2 service ticket: %s (error code %s)', e.message, e.status)

        if self._user_context and tvm_2_user_ticket:
            try:
                user_ticket = self._user_context.check(tvm_2_user_ticket)
            except TicketParsingException as e:
                logger.debug('Bad TVM2 user ticket: %s (error code %s)', e.message, e.status)

        return TicketCheckResult(service_ticket, user_ticket)

    def check_headers(self, headers):
        """Given headers mapping check TVM tickets in them.
        Args:
            headers (mapping): headers with ticket values, possible headers:
                ticket: contains TVM 1.0 service ticket
                x-ya-user-ticket: contains TVM 2.0 user ticket
                x-ya-service-ticket: contains TVM 2.0 service ticket
        Returns:
            TicketCheckResult
        """
        self._refresh_context()

        tvm_headers = dict(
            (k.lower(), self._fix_bytes(v))
            for k, v in headers.items()
            if k.lower() in ('x-ya-service-ticket', 'x-ya-user-ticket')
        )

        return self.check_tickets(
            tvm_2_service_ticket=tvm_headers.get('x-ya-service-ticket'),
            tvm_2_user_ticket=tvm_headers.get('x-ya-user-ticket'),
        )

    def _init_contexts(self):
        """Initialize context objects to check tickets."""
        self._service_context = tvm2.ServiceContext(
            self._tvm.client_id,
            self._tvm.secret_v2,
            self._tvm.tvm2_keys
        )

        self._user_context = None
        self._legacy_context = None
        if self._tvm.use_passport:
            self._user_context = tvm2.UserContext(
                self._tvm.passport_env,
                self._tvm.tvm2_keys,
            )

        self._keys_loaded_at = self._tvm.keys_loaded_at

    @staticmethod
    def _fix_bytes(value):
        if isinstance(value, six.binary_type):
            return value
        return value.encode('utf-8')


class TicketCheckResult(object):
    """
        Результат проверки токенов из заголовков.
        Интерпретируются заголовки обоих версий TVM.
        TVM2 идентифицируется сервисным токеном.
    """

    def __init__(self, service, user):
        self._service_ticket = service
        self._user_ticket = user

    @property
    def valid(self):
        """ У нас есть хотя бы один сервисный токен """
        return self._service_ticket is not None

    @property
    def legacy(self):
        """ Есть только легаси токен """
        return False

    @property
    def src(self):
        """ От кого пришел токен """
        if self._service_ticket:
            return self._service_ticket.src
        return None

    @property
    def scopes(self):
        """ Скоупы токена. На текущий момент не реализованы в TVM """
        # TODO: у пользовательского токена тоже есть скоупы, но пока это не работает
        if self._service_ticket:
            return tuple(self._service_ticket.scopes)
        return ()

    @property
    def uids(self):
        """ Набор uid'ов из пользовательского токена. Для легаси - только safe_uids"""
        if self._service_ticket and self._user_ticket:
            return self._user_ticket.uids
        return ()

    @property
    def default_uid(self):
        """ uid по-умолчанию """
        if self._service_ticket and self._user_ticket:
            return self._user_ticket.default_uid
        return None

    def has_src(self, src):
        src = int(src)
        if self._service_ticket:
            return self._service_ticket.src == src
        return False

    def has_scope(self, scope):
        if self._service_ticket:
            return self._service_ticket.has_scope(scope)
        return False

    def has_uid(self, uid):
        if self._service_ticket and self._user_ticket:
            return uid in self._user_ticket.uids
        return False
