#pragma once
// DO_NOT_STYLE

#include <stddef.h>
#include <stdint.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

enum TA_EErrorCode {
    TA_EC_OK = 0,
    TA_EC_DEPRECATED,
    TA_EC_EMPTY_TVM_KEYS,
    TA_EC_EXPIRED_TICKET,
    TA_EC_INVALID_BLACKBOX_ENV,
    TA_EC_INVALID_DST,
    TA_EC_INVALID_PARAM,
    TA_EC_INVALID_TICKET_TYPE,
    TA_EC_MALFORMED_TICKET,
    TA_EC_MALFORMED_TVM_KEYS,
    TA_EC_MALFORMED_TVM_SECRET,
    TA_EC_MISSING_KEY,
    TA_EC_NOT_ALLOWED,
    TA_EC_SIGN_BROKEN,
    TA_EC_SMALL_BUFFER,
    TA_EC_UNEXPECTED_ERROR,
    TA_EC_UNSUPPORTED_VERSION,
};
struct TA_TServiceContext;
struct TA_TCheckedServiceTicket;
struct TA_TUserContext;
struct TA_TCheckedUserTicket;

extern const char* TA_BlackboxClientIdProd;       /* 222 */
extern const char* TA_BlackboxClientIdTest;       /* 224 */
extern const char* TA_BlackboxClientIdProdYateam; /* 223 */
extern const char* TA_BlackboxClientIdTestYateam; /* 225 */
extern const char* TA_BlackboxClientIdStress;     /* 226 */
extern const char* TA_BlackboxClientIdMimino;     /* 239 */

enum TA_EBlackboxEnv {
    TA_BE_PROD = 0,
    TA_BE_TEST,
    TA_BE_PROD_YATEAM,
    TA_BE_TEST_YATEAM,
    TA_BE_STRESS
};

const char* TA_ErrorCodeToString(enum TA_EErrorCode code);

/*!
 * Create service context. Serivce contexts are used to store TVM keys and parse service tickets.
 * @param[in] clientId
 * @param[in] secretBase64 Secret, can be NULL. Sign attempt hence raises exception
 * @param[in] secretBase64Size Size of string containing secret
 * @param[in] tvmKeysResponse TVM Keys gotten from TVM API
 * @param[in] tvmKeysResponseSize Size of string containing keys
 * @param[out] context
 * @return Error code
 */
enum TA_EErrorCode TA_CreateServiceContext(
    uint32_t clientId,
    const char* secretBase64,
    size_t secretBase64Size,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize,
    struct TA_TServiceContext** context);

/*!
 * Free memory of service context.
 * @param[in] context
 * @return Error code
 */
enum TA_EErrorCode TA_DeleteServiceContext(struct TA_TServiceContext* context);

/*!
 * Parse and validate service ticket body then create TServiceTicket object.
 * @param[in] context
 * @param[in] ticketBody Service ticket body as string
 * @param[in] ticketBodySize Size of string containing service ticket body
 * @param[out] ticket Service ticket object
 * @return Error code
 */
enum TA_EErrorCode TA_CheckServiceTicket(
    const struct TA_TServiceContext* context,
    const char* ticketBody,
    size_t ticketBodySize,
    struct TA_TCheckedServiceTicket** ticket);

/*!
 * Free memory of service ticket.
 * @param[in] ticket
 * @return Error code
 */
enum TA_EErrorCode TA_DeleteServiceTicket(struct TA_TCheckedServiceTicket* ticket);

/*!
 * Source of request, your client.
 * @param[in] ticket Parsed ticket
 * @param[out] srcClientId Integer identifier of client
 * @return Error code
 */
enum TA_EErrorCode TA_GetServiceTicketSrc(
    const struct TA_TCheckedServiceTicket* ticket,
    uint32_t* srcClientId);

/*!
 * Return debug info for ticket
 * @param[in] ticket
 * @param[out] debugInfo
 * @param[out] debugInfoSize
 * @return Error code
 * @param[in] maxDebugInfoSize
 */
enum TA_EErrorCode TA_GetServiceTicketDebugInfo(
    const struct TA_TCheckedServiceTicket* ticket,
    char* debugInfo,
    size_t* debugInfoSize,
    size_t maxDebugInfoSize);

/*!
 * Return part of ticket which can be safely logged.
 * WARNING: Do not free returned pointer.
 * WARNING: logableTicket is not zero-ended string (it is substring of ticketBody)
 * WARNING: Use logableTicketSize to read valid amount of symbols.
 * @param[in] ticketBody
 * @param[in] ticketBodySize
 * @param[out] logableTicket
 * @param[out] logableTicketSize
 * @return Error code
 */
enum TA_EErrorCode TA_RemoveServiceTicketSignature(
    const char* ticketBody,
    size_t ticketBodySize,
    const char** logableTicket,
    size_t* logableTicketSize);

/*!
 * Create signature for selected parameters. Allocate at least 512 byte for signature buffer.
 * @param[in] context
 * @param[in] ts Param 'ts' of request to TVM
 * @param[in] tsSize Size of param 'ts' of request to TVM
 * @param[in] dst Param 'dst' of request to TVM
 * @param[in] dstSize Size of param 'dst' of request to TVM
 * @param[in] scopes Param 'scopes' of request to TVM
 * @param[in] scopesSize Size of param 'scopes' of request to TVM
 * @param[out] signature
 * @param[out] signatureSize
 * @param[in] maxSignatureSize
 * @return Error code
 */
enum TA_EErrorCode TA_SignCgiParamsForTvm(
    const struct TA_TServiceContext* context,
    const char* ts,
    size_t tsSize,
    const char* dst,
    size_t dstSize,
    const char* scopes,
    size_t scopesSize,
    char* signature,
    size_t* signatureSize,
    size_t maxSignatureSize);

/*!
 * Create user context. User contexts are used to store TVM keys and parse user tickets.
 * @param[in] env
 * @param[in] tvmKeysResponse
 * @param[in] tvmKeysResponseSize
 * @param[out] context
 * @return Error code
 */
enum TA_EErrorCode TA_CreateUserContext(
    enum TA_EBlackboxEnv env,
    const char* tvmKeysResponse,
    size_t tvmKeysResponseSize,
    struct TA_TUserContext** context);

/*!
 * Free memory of user context.
 * @param[in] context
 * @return Error code
 */
enum TA_EErrorCode TA_DeleteUserContext(struct TA_TUserContext* context);

/*!
 * Parse and validate user ticket body then create TUserTicket object.
 * @param[in] context
 * @param[in] ticketBody Service ticket body as string
 * @param[in] ticketBodySize Size of string containing service ticket body
 * @param[out] ticket Service ticket object
 * @return Error code
 */
enum TA_EErrorCode TA_CheckUserTicket(
    const struct TA_TUserContext* context,
    const char* ticketBody,
    size_t ticketBodySize,
    struct TA_TCheckedUserTicket** ticket);

/*!
 * Free memory of user ticket.
 * @param[in] context
 * @return Error code
 */
enum TA_EErrorCode TA_DeleteUserTicket(struct TA_TCheckedUserTicket* context);

/*!
 * Return number of UIDs in the ticket.
 * @param[in] ticket
 * @param[out] count
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketUidsCount(
    const struct TA_TCheckedUserTicket* ticket,
    size_t* count);

/*!
 * Return UID from ticket by ordinal index
 * @param[in] ticket
 * @param[in] idx
 * @param[out] uid
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketUid(
    const struct TA_TCheckedUserTicket* ticket,
    size_t idx,
    uint64_t* uid);

/*!
 * Return default UID. Default UID is the chosen one which should be considered as primary.
 * @param[in] ticket
 * @param[out] uid
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketDefaultUid(
    const struct TA_TCheckedUserTicket* ticket,
    uint64_t* uid);

/*!
 * Return number of scopes in the ticket.
 * @param[in] ticket
 * @param[out] count
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketScopesCount(
    const struct TA_TCheckedUserTicket* ticket,
    size_t* count);

/*!
 * Return scope by ordinal index.
 * WARNING: Do not free returned pointer.
 * @param[in] ticket
 * @param[in] idx
 * @param[out] scope
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketScope(
    const struct TA_TCheckedUserTicket* ticket,
    size_t idx,
    const char** scope);

/*!
 * Check if user ticket has the scope
 * @param[in] ticket
 * @param[in] scope
 * @param[in] scopeSize Size of string containing scope
 * @param[out] checkingResult Equal to 1 if scope is present in ticket and to 0 otherwise
 * @return Error code
 */
enum TA_EErrorCode TA_HasUserTicketScope(
    const struct TA_TCheckedUserTicket* ticket,
    const char* scope,
    size_t scopeSize,
    int* checkingResult);

/*!
 * Return debug info for ticket
 * @param[in] ticket
 * @param[out] debugInfo
 * @param[out] debugInfoSize
 * @param[in] maxDebugInfoSize
 * @return Error code
 */
enum TA_EErrorCode TA_GetUserTicketDebugInfo(
    const struct TA_TCheckedUserTicket* ticket,
    char* debugInfo,
    size_t* debugInfoSize,
    size_t maxDebugInfoSize);

/*!
 * Return part of ticket which can be safely logged.
 * WARNING: Do not free returned pointer.
 * WARNING: logableTicket is not zero-ended string (it is substring of ticketBody)
 * WARNING: Use logableTicketSize to read valid amount of symbols.
 * @param[in] ticketBody
 * @param[in] ticketBodySize
 * @param[out] logableTicket
 * @param[out] logableTicketSize
 * @return Error code
 */
enum TA_EErrorCode TA_RemoveUserTicketSignature(
    const char* ticketBody,
    size_t ticketBodySize,
    const char** logableTicket,
    size_t* logableTicketSize);

/*!
 * Return library version.
 * @return version
 */
const char* TA_LibVersion();

#ifdef __cplusplus
}
#endif
