import six

from enum import Enum, IntEnum
from ..common.exceptions import (
    TvmException,
    ContextException,
    TicketException,
    TicketParsingException,
)

class BlackboxEnv(IntEnum):
        PROD = TA_BE_PROD
        TEST = TA_BE_TEST
        PROD_YATEAM = TA_BE_PROD_YATEAM
        TEST_YATEAM = TA_BE_TEST_YATEAM
        STRESS = TA_BE_STRESS

class BlackboxClientId(Enum):
    PROD = b'222'
    TEST = b'224'
    PROD_YATEAM = b'223'
    TEST_YATEAM = b'225'
    STRESS = b'226'
    MIMINO = b'239'

def error_code_to_string(code):
    return TA_ErrorCodeToString(code)

cdef check_ok(code, error_cls=TvmException):
    if code == TA_EC_OK:
        return
    raise error_cls(error_code_to_string(code))

def version():
    cdef const char*  version
    version = TA_LibVersion()
    return version

def force_bytes(value):
    if isinstance(value, six.binary_type):
        return value
    if isinstance(value, int):
        value = six.text_type(value)
    if isinstance(value, six.text_type):
        return value.encode('utf-8')
    raise TypeError

def force_ts_bytes(value):
    if isinstance(value, float):
        value = int(value)
    return force_bytes(value)


cdef class ServiceContext(object):
    cdef TA_TServiceContext* ctx

    def __init__(self, client_id, secret, tvm_keys):
        check_ok(TA_CreateServiceContext(client_id, secret, len(secret), tvm_keys, len(tvm_keys), &self.ctx), error_cls=ContextException)

    def __dealloc__(self):
        check_ok(TA_DeleteServiceContext(self.ctx))

    cdef __sign(self, timestamp, dst, scopes):
        cdef char buf[1024]
        cdef size_t buf_len
        check_ok(TA_SignCgiParamsForTvm(self.ctx, timestamp, len(timestamp), dst, len(dst), scopes, len(scopes), buf, &buf_len, 1024))
        return buf[:buf_len]

    def check(self, ticket_body):
        ticket = ServiceTicket()
        ticket_body = force_bytes(ticket_body)

        cdef TA_TCheckedServiceTicket* t
        status = TA_CheckServiceTicket(self.ctx, ticket_body, len(ticket_body), &t)
        ticket.t = t
        if status != TA_EC_OK:
            raise TicketParsingException(error_code_to_string(status), status, ticket.debug_info())
        return ticket

    def sign(self, timestamp, dst, scopes=None):
        if isinstance(dst, list):
            dst = b','.join(map(lambda x: x if isinstance(x, six.binary_type) else six.text_type(x).encode('utf-8'), dst))
        if isinstance(scopes, list):
            scopes = b','.join(map(lambda x: x if isinstance(x, six.binary_type) else six.text_type(x).encode('utf-8'), scopes))
        elif scopes is None:
            scopes = b''
        return self.__sign(force_ts_bytes(timestamp), force_bytes(dst), force_bytes(scopes))


cdef class ServiceTicket(object):
    cdef TA_TCheckedServiceTicket* t

    def __dealloc__(self):
        check_ok(TA_DeleteServiceTicket(self.t))

    def __str__(self):
        return self.debug_info().decode('utf-8')

    def __repr__(self):
        return str(self)

    def debug_info(self):
        cdef char debug_info[1024]
        cdef size_t debug_info_len
        check_ok(TA_GetServiceTicketDebugInfo(self.t, debug_info, &debug_info_len, 1024), error_cls=TicketException)
        return debug_info[:debug_info_len]

    @property
    def src(self):
        cdef uint32_t client_id
        check_ok(TA_GetServiceTicketSrc(self.t, &client_id), error_cls=TicketException)
        return client_id


cdef class UserContext:
    cdef TA_TUserContext* ctx
    def __init__(self, env_, tvm_keys):
        cdef TA_EBlackboxEnv env
        if isinstance(env_, six.text_type):
            env = <TA_EBlackboxEnv>(BlackboxEnv[env_].value)
        else:
            env = <TA_EBlackboxEnv>(env_.value)
        check_ok(TA_CreateUserContext(env, tvm_keys, len(tvm_keys), &self.ctx), error_cls=ContextException)

    def __dealloc__(self):
        check_ok(TA_DeleteUserContext(self.ctx))

    def check(self, ticket_body):
        ticket = UserTicket()
        ticket_body = force_bytes(ticket_body)

        cdef TA_TCheckedUserTicket* t
        status = TA_CheckUserTicket(self.ctx, ticket_body, len(ticket_body), &t)
        ticket.t = t
        if status != TA_EC_OK:
            raise TicketParsingException(error_code_to_string(status), status, ticket.debug_info())
        return ticket


cdef class UserTicket:
    cdef TA_TCheckedUserTicket* t

    def __dealloc__(self):
        check_ok(TA_DeleteUserTicket(self.t))

    def __str__(self):
        return self.debug_info().decode('utf-8')

    def __repr__(self):
        return str(self)

    def debug_info(self):
        cdef char debug_info[1024]
        cdef size_t debug_info_len
        check_ok(TA_GetUserTicketDebugInfo(self.t, debug_info, &debug_info_len, 1024), error_cls=TicketException)
        return debug_info[:debug_info_len]

    @property
    def default_uid(self):
        cdef uint64_t uid
        check_ok(TA_GetUserTicketDefaultUid(self.t, &uid), error_cls=TicketException)
        return uid

    def has_scope(self, scope_name):
        cdef int result
        scope_name = force_bytes(scope_name)
        check_ok(TA_HasUserTicketScope(self.t, scope_name, len(scope_name), &result), error_cls=TicketException)
        return bool(result)

    @property
    def scopes(self):
        result = []
        cdef size_t scopes_cnt
        cdef const char* scope

        check_ok(TA_GetUserTicketScopesCount(self.t, &scopes_cnt), error_cls=TicketException)
        for idx in range(scopes_cnt):
            check_ok(TA_GetUserTicketScope(self.t, idx, &scope), error_cls=TicketException)
            result.append(scope[:])
        return result

    @property
    def uids(self):
        result = []
        cdef size_t uids_count
        cdef uint64_t uids

        check_ok(TA_GetUserTicketUidsCount(self.t, &uids_count), error_cls=TicketException)
        for idx in range(uids_count):
            check_ok(TA_GetUserTicketUid(self.t, idx, &uids), error_cls=TicketException)
            result.append(uids)
        return result
